<?php

namespace App\Console\Commands;

use App\Mail\HoldReleasedMail;
use App\Models\Deposit;
use Illuminate\Console\Command;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Stripe\Exception\ApiErrorException;
use Stripe\StripeClient;

class AutoCancelHolds extends Command
{
    protected $signature = 'holds:auto-cancel
        {--dry : Don\'t call Stripe or write to DB}
        {--limit= : Optional hard limit of rows to process this run}';

    protected $description = 'Release (cancel) uncaptured manual-capture holds when planned time arrives (clamped by Stripe auth ceiling).';

    public function handle(): int
    {
        $now         = now();
        $dry         = (bool) $this->option('dry');
        $batchSize   = (int) (config('holds.batch_size', 200) ?: 200);
        $lockSeconds = (int) (config('holds.lock_seconds', 90) ?: 90);
        $hardLimit   = $this->option('limit') ? (int) $this->option('limit') : null;

        $lock = Cache::lock('holds:auto-cancel-running', $lockSeconds);
        if (! $lock->get()) {
            $this->warn('Another auto-cancel run appears to be in progress. Exiting.');
            return self::SUCCESS;
        }

        try {
            $stripe = new StripeClient((string) config('services.stripe.secret'));

            // Authorised holds that still have a PI to cancel.
            $q = Deposit::query()
                ->with(['job', 'job.flow', 'job.brand', 'job.vehicle'])
                ->whereIn('status', ['authorized', 'authorised'])
                ->whereNotNull('stripe_payment_intent')
                ->orderBy('id');

            $total = (int) $q->count();
            $this->info(sprintf(
                'Auto-cancel starting at %s (dry=%s, batch=%d%s). Scanning %d deposits…',
                $now->toDateTimeString(),
                $dry ? 'yes' : 'no',
                $batchSize,
                $hardLimit ? ", limit={$hardLimit}" : '',
                $total
            ));

            $bar       = $this->output->createProgressBar($total);
            $handled   = 0;
            $skipped   = 0;
            $errors    = 0;
            $processed = 0;

            $q->chunkById($batchSize, function ($deposits) use ($now, $stripe, $dry, $hardLimit, $bar, &$handled, &$skipped, &$errors, &$processed) {
                /** @var \App\Models\Deposit $deposit */
                foreach ($deposits as $deposit) {
                    if ($hardLimit !== null && $processed >= $hardLimit) {
                        return false; // stop chunking
                    }
                    $processed++;

                    [$planned, $ceiling, $effectiveDue] = $this->computeDueTimes($deposit);

                    // If no due time could be computed, skip.
                    if (! $effectiveDue) {
                        $skipped++;
                        Log::warning('AutoCancelHolds: no effective due time; skipping', [
                            'deposit_id' => $deposit->id,
                            'job_id'     => optional($deposit->job)->id,
                            'pi'         => $deposit->stripe_payment_intent,
                        ]);
                        $bar->advance();
                        continue;
                    }

                    // Not yet due.
                    if ($now->lt($effectiveDue)) {
                        $skipped++;
                        $bar->advance();
                        continue;
                    }

                    // Due — cancel/release the hold.
                    try {
                        if ($dry) {
                            Log::info('AutoCancelHolds (dry): would cancel hold', [
                                'deposit_id' => $deposit->id,
                                'job_id'     => optional($deposit->job)->id,
                                'pi'         => $deposit->stripe_payment_intent,
                                'planned_at' => $planned?->toIso8601String(),
                                'ceiling_at' => $ceiling?->toIso8601String(),
                                'due_at'     => $effectiveDue->toIso8601String(),
                            ]);
                            $handled++;
                            $bar->advance();
                            continue;
                        }

                        DB::transaction(function () use ($stripe, $deposit, $planned, $ceiling, $effectiveDue, &$handled) {
                            // Try to cancel the Stripe authorization (PI)
                            try {
                                $this->cancelStripeAuthorization($deposit, $stripe);
                            } catch (\Throwable $e) {
                                // Log but continue: PI may already be canceled/expired; we still mark released locally.
                                Log::warning('Stripe PI cancel error (continuing): ' . $e->getMessage(), [
                                    'deposit_id' => $deposit->id,
                                    'pi'         => $deposit->stripe_payment_intent,
                                ]);
                            }

                            // Persist state
                            $deposit->status = 'released'; // (or 'canceled' if your UI expects that)

                            $table = $deposit->getTable();
                            if (Schema::hasColumn($table, 'released_at') && empty($deposit->released_at)) {
                                $deposit->released_at = now();
                            }
                            if (Schema::hasColumn($table, 'last_reason')) {
                                $deposit->last_reason = 'auto-cancel';
                            }

                            // Minimal audit trail
                            $meta = (array) ($deposit->meta ?? []);
                            $meta['auto_cancel'] = [
                                'planned_cancel_at' => $planned?->toIso8601String(),
                                'ceiling_at'        => $ceiling?->toIso8601String(),
                                'effective_due_at'  => $effectiveDue?->toIso8601String(),
                                'released_at'       => now()->toIso8601String(),
                                'reason'            => 'auto_cancel_if_no_capture',
                            ];
                            $deposit->meta = $meta;

                            $deposit->save();

                            // Email after commit (queued)
                            DB::afterCommit(function () use ($deposit) {
                                try {
                                    $job = $deposit->job;
                                    $to  = $job?->customer_email;

                                    if (! $to) {
                                        return; // no email on file
                                    }

                                    Mail::to($to)->queue(new HoldReleasedMail($deposit, $job));
                                } catch (\Throwable $mailEx) {
                                    Log::warning('AutoCancelHolds: email queue failed', [
                                        'deposit_id' => $deposit->id,
                                        'error'      => $mailEx->getMessage(),
                                    ]);
                                }
                            });

                            $handled++;
                        });
                    } catch (\Throwable $ex) {
                        $errors++;
                        Log::error('AutoCancelHolds: failure canceling hold', [
                            'deposit_id' => $deposit->id,
                            'job_id'     => optional($deposit->job)->id,
                            'pi'         => $deposit->stripe_payment_intent,
                            'error'      => $ex->getMessage(),
                        ]);
                    } finally {
                        $bar->advance();
                    }
                }
            });

            $bar->finish();
            $this->newLine(2);
            $this->info("Done. handled={$handled}, skipped={$skipped}, errors={$errors}");
        } finally {
            optional($lock)->release();
        }

        return self::SUCCESS;
    }

    /**
     * Compute when a hold should be released.
     *
     * Returns: [Carbon|null $planned, Carbon|null $ceiling, Carbon|null $effective]
     * - planned: explicit planned_cancel_at, or derived fallback(s)
     * - ceiling: hard ceiling from authorized_at + Stripe auth window days
     * - effective: min(planned, ceiling) (ignores nulls gracefully)
     */
    protected function computeDueTimes(Deposit $deposit): array
    {
        $defaultMinutes = (int) (config('holds.default_cancel_after_minutes', 720) ?: 720);
        $authWindowDays = (int) (config('holds.stripe_auth_window_days', 7) ?: 7);

        // 1) Explicit plan on deposit
        $planned = $this->toCarbonOrNull($deposit->planned_cancel_at);

        // 2) Fallback: job end time + config minutes
        if (! $planned && $deposit->relationLoaded('job') && $deposit->job && $deposit->job->end_at) {
            $planned = Carbon::parse($deposit->job->end_at)->addMinutes($defaultMinutes);
        }

        // 3) Fallback: from Job/Flow if available
        if (! $planned) {
            $planned = $this->computePlannedFromJobAndFlow(optional($deposit->job));
        }

        // 4) Final fallback if still empty: authorized_at + default minutes
        if (! $planned && $deposit->authorized_at) {
            $authAt  = $this->toCarbonOrNull($deposit->authorized_at);
            $planned = $authAt ? $authAt->copy()->addMinutes($defaultMinutes) : null;
        }

        // Stripe auth ceiling (authorized_at + N days)
        $ceiling = null;
        if (! empty($deposit->authorized_at)) {
            $authAt = $this->toCarbonOrNull($deposit->authorized_at);
            if ($authAt) {
                $ceiling = $authAt->copy()->addDays($authWindowDays);
            }
        }

        // Effective due time = min(planned, ceiling) that exist
        $effective = null;
        if ($planned && $ceiling) {
            $effective = $planned->min($ceiling);
        } elseif ($planned) {
            $effective = $planned;
        } elseif ($ceiling) {
            $effective = $ceiling;
        }

        return [$planned, $ceiling, $effective];
    }

    /**
     * Legacy-friendly derivation from Job and/or Flow if available.
     */
    protected function computePlannedFromJobAndFlow($job): ?Carbon
    {
        if (! $job) {
            return null;
        }

        // If Job exposes a helper, use it.
        if (method_exists($job, 'computePlannedCancelAt')) {
            $c = $job->computePlannedCancelAt();
            if ($c instanceof Carbon) {
                return $c;
            }
        }

        // Try derive from Flow fields if relation exists
        if (isset($job->flow) && $job->flow) {
            $minutes = $this->resolveFlowCancelMinutes(
                $job->flow,
                (int) (config('holds.default_cancel_after_minutes', 720) ?: 720)
            );
            if (! empty($job->end_at)) {
                return Carbon::parse($job->end_at)->addMinutes($minutes);
            }
        }

        return null;
    }

    /**
     * Resolve minutes value from various possible Flow field names (legacy tolerance).
     */
    protected function resolveFlowCancelMinutes($flow, int $defaultMinutes = 720): int
    {
        $candidates = [
            // minutes
            'auto_cancel_after_minutes',
            'hold_auto_cancel_after_minutes',
            'deposit_auto_cancel_after_minutes',
            'auto_cancel_minutes',
            'hold_auto_cancel_minutes',
            'deposit_auto_cancel_minutes',
            // hours
            'auto_cancel_after_hours',
            'hold_auto_cancel_after_hours',
            'deposit_auto_cancel_after_hours',
            'auto_cancel_hours',
            'hold_auto_cancel_hours',
            'deposit_auto_cancel_hours',
            // days
            'auto_cancel_after_days',
            'hold_auto_cancel_after_days',
            'deposit_auto_cancel_after_days',
            'auto_cancel_days',
            'hold_auto_cancel_days',
            'deposit_auto_cancel_days',
        ];

        $value = null;
        $unit  = 'm';

        foreach ($candidates as $key) {
            $exists = is_array($flow) ? array_key_exists($key, $flow) : (is_object($flow) && isset($flow->{$key}));
            if ($exists) {
                $v = is_array($flow) ? ($flow[$key] ?? null) : ($flow->{$key} ?? null);
                if ($v !== null && $v !== '') {
                    $value = (int) $v;
                    $unit  = str_ends_with($key, '_minutes') ? 'm'
                          : (str_ends_with($key, '_hours')   ? 'h'
                          : (str_ends_with($key, '_days')    ? 'd' : 'm'));
                    break;
                }
            }
        }

        if ($value === null) {
            return $defaultMinutes;
        }

        return match ($unit) {
            'm' => $value,
            'h' => $value * 60,
            'd' => $value * 60 * 24,
            default => $defaultMinutes,
        };
    }

    protected function toCarbonOrNull($value): ?Carbon
    {
        if ($value instanceof Carbon) {
            return $value;
        }
        if (empty($value)) {
            return null;
        }
        try {
            return Carbon::parse($value);
        } catch (\Throwable) {
            return null;
        }
    }

    /**
     * Cancel (release) the uncaptured authorization by canceling the PaymentIntent.
     * Kept as a protected instance method so tests can Mockery::mock()->shouldReceive(...) easily.
     */
    protected function cancelStripeAuthorization(Deposit $deposit, ?StripeClient $stripe = null): bool
    {
        $piId = (string) $deposit->stripe_payment_intent;
        if ($piId === '') {
            // nothing to cancel
            return true;
        }

        $stripe ??= new StripeClient((string) config('services.stripe.secret'));

        try {
            $pi = $stripe->paymentIntents->cancel($piId, []);
            return in_array($pi->status, ['canceled', 'cancelled'], true);
        } catch (ApiErrorException $e) {
            throw $e;
        }
    }
}
