<?php

declare(strict_types=1);

namespace App\Console\Commands;

use App\Models\Job;
use App\Models\Owner;
use App\Models\OwnerLedgerEntry;
use App\Models\Vehicle;
use App\Models\VehicleMaintenanceLog;
use Illuminate\Console\Command;

class BackfillOwnerLedger extends Command
{
    protected $signature = 'owners:backfill-ledger {--dry-run : Show what would be written without saving}';

    protected $description = 'Backfill owner earnings from jobs and costs from maintenance logs.';

    public function handle(): int
    {
        $dryRun = (bool) $this->option('dry-run');

        $this->info($dryRun
            ? 'Running in DRY RUN mode – no changes will be saved.'
            : 'Backfilling owner ledger entries...'
        );
        $this->newLine();

        $this->line('Job counts:');
        $this->line('  with scopes:      ' . Job::count());
        $this->line('  without scopes:   ' . Job::withoutGlobalScopes()->count());
        $this->line('Maintenance counts:');
        $this->line('  with scopes:      ' . VehicleMaintenanceLog::count());
        $this->line('  without scopes:   ' . VehicleMaintenanceLog::withoutGlobalScopes()->count());
        $this->newLine();

        $earningsCount = $this->backfillJobEarnings($dryRun);
        $costsCount    = $this->backfillMaintenanceCosts($dryRun);

        $this->newLine();
        $this->info('Done.');
        $this->info("Earnings rows written: {$earningsCount}");
        $this->info("Cost rows written:     {$costsCount}");

        if ($dryRun) {
            $this->warn('No DB changes were made (dry-run). Run again without --dry-run to persist.');
        }

        return self::SUCCESS;
    }

    protected function backfillJobEarnings(bool $dryRun = false): int
{
    $this->info('Backfilling earnings from jobs...');

    $written = 0;

    $jobs = Job::withoutGlobalScopes()
        ->whereNotNull('vehicle_id')
        ->get();

    $this->line('  Jobs with vehicle_id not null: ' . $jobs->count());

    foreach ($jobs as $job) {
        $vehicle = Vehicle::withoutGlobalScopes()->find($job->vehicle_id);
        $owner   = $vehicle
            ? Owner::withoutGlobalScopes()->find($vehicle->owner_id)
            : null;

        $gross = (int) ($job->charge_amount_cents ?? 0);

        // Log some context
        $this->line(sprintf(
            '  Job %d: vehicle_id=%s, owner_id=%s, gross_cents=%d',
            $job->id,
            $job->vehicle_id ?? 'null',
            $vehicle?->owner_id ?? 'null',
            $gross,
        ));

        if (! $owner || $gross <= 0) {
            continue;
        }

        // ---- OWNER SPLIT HERE ----
        // Use vehicle->owner_share_percent if set, otherwise default to 70%
        $sharePercent = $vehicle->owner_share_percent ?? 70;
        $sharePercent = max(0, min(100, (int) $sharePercent));

        $amount = (int) floor($gross * $sharePercent / 100);

        if ($amount <= 0) {
            $this->line(sprintf(
                '    -> Skipping: share %d%% gives amount 0',
                $sharePercent,
            ));
            continue;
        }

        $occurredAt = $job->end_at
            ?? $job->start_at
            ?? $job->created_at
            ?? now();

        $this->line(sprintf(
            '    -> [EARNING] Owner %d, Vehicle %d, Job %d, Share %d%%, Amount %0.2f',
            $owner->id,
            $vehicle->id,
            $job->id,
            $sharePercent,
            $amount / 100,
        ));

        if (! $dryRun) {
            OwnerLedgerEntry::updateOrCreate(
                [
                    'owner_id' => $owner->id,
                    'job_id'   => $job->id,
                    'type'     => 'earning',
                ],
                [
                    'brand_id'     => $owner->brand_id ?? null,
                    'vehicle_id'   => $vehicle->id,
                    'amount_cents' => $amount,
                    'occurred_at'  => $occurredAt,
                    'description'  => $job->booking_reference ?: 'Hire revenue',
                ],
            );
        }

        $written++;
    }

    return $written;
}

    protected function backfillMaintenanceCosts(bool $dryRun = false): int
    {
        $this->info('Backfilling costs from maintenance logs...');

        $written = 0;

        $logs = VehicleMaintenanceLog::withoutGlobalScopes()
            ->whereNotNull('owner_cost_cents')
            ->where('owner_cost_cents', '>', 0)
            ->get();

        $this->line('  Logs with owner_cost_cents > 0: ' . $logs->count());

        foreach ($logs as $log) {
            $vehicle = Vehicle::withoutGlobalScopes()->find($log->vehicle_id);
            $owner   = $vehicle
                ? Owner::withoutGlobalScopes()->find($vehicle->owner_id)
                : null;

            $amount = (int) $log->owner_cost_cents;

            $this->line(sprintf(
                '  Log %d: vehicle_id=%s, owner_id=%s, owner_cost_cents=%d',
                $log->id,
                $log->vehicle_id ?? 'null',
                $vehicle?->owner_id ?? 'null',
                $amount,
            ));

            if (! $owner || $amount <= 0) {
                continue;
            }

            $occurredAt  = $log->performed_at ?? $log->created_at ?? now();
            $description = $log->owner_cost_description
                ?: ($log->title ?: 'Maintenance cost');

            $this->line(sprintf(
                '    -> [COST] Owner %d, Vehicle %d, Log %d, Amount %0.2f',
                $owner->id,
                $vehicle->id,
                $log->id,
                $amount / 100,
            ));

            if (! $dryRun) {
                $entry = OwnerLedgerEntry::updateOrCreate(
                    [
                        'owner_id'           => $owner->id,
                        'maintenance_log_id' => $log->id,
                        'type'               => 'cost',
                    ],
                    [
                        'brand_id'     => $owner->brand_id ?? null,
                        'vehicle_id'   => $vehicle->id,
                        'job_id'       => null,
                        'amount_cents' => $amount,
                        'occurred_at'  => $occurredAt,
                        'description'  => $description,
                    ],
                );

                if ($log->owner_ledger_entry_id !== $entry->id) {
                    $log->owner_ledger_entry_id = $entry->id;
                    $log->saveQuietly();
                }
            }

            $written++;
        }

        return $written;
    }
}
