<?php

namespace App\Console\Commands;

use App\Models\Booking;
use App\Models\VevsVehicle;
use Illuminate\Console\Command;
use Throwable;

class VeVsSyncCars extends Command
{
    protected $signature = 'vevs:sync-cars
        {--limit=0 : Max distinct vevs_car_id values to process (0 = no limit)}
        {--dump : Dump the first derived vehicle for debugging}';

    protected $description = 'Derive VEVS vehicle list from existing bookings (vevs_car_id + vehicle label)';

    public function handle(): int
    {
        $limit = (int) ($this->option('limit') ?? 0);
        $dump  = (bool) $this->option('dump');

        $this->info('Deriving vehicle list from bookings…');

        // Grab bookings with a VEVS car id
        $query = Booking::query()
            ->whereNotNull('vevs_car_id')
            ->orderBy('vevs_car_id')
            ->orderByDesc('id'); // latest booking per car first

        $bookings = $query->get();

        if ($bookings->isEmpty()) {
            $this->warn('No bookings with vevs_car_id found.');
            return Command::SUCCESS;
        }

        // Group by vevs_car_id so we can derive a single entry per car
        $grouped = $bookings->groupBy('vevs_car_id');

        if ($limit > 0) {
            $grouped = $grouped->slice(0, $limit);
        }

        $total   = $grouped->count();
        $success = 0;
        $failed  = 0;

        $this->info("Processing {$total} distinct vevs_car_id value(s)…");
        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $firstDumped = false;

        foreach ($grouped as $vevsCarId => $rows) {
            try {
                // Pick a representative booking for this car id (latest)
                /** @var \App\Models\Booking $sample */
                $sample = $rows->first();

                $name = $sample->vehicle ?: null; // we just wired this from VEVS "type"
                $label = $name;                   // you can customize later if you like

                $raw = [
                    'sample_booking_id'    => $sample->id,
                    'sample_reference'     => $sample->reference,
                    'sample_vehicle_field' => $sample->vehicle,
                ];

                VevsVehicle::updateOrCreate(
                    ['vevs_car_id' => (string) $vevsCarId],
                    [
                        'name'  => $name,
                        'label' => $label,
                        'type'  => null, // spare column for future use
                        'raw'   => $raw,
                    ]
                );

                if ($dump && ! $firstDumped) {
                    $this->newLine(2);
                    $this->line('--- Sample derived vehicle ---');
                    $this->line('vevs_car_id: ' . $vevsCarId);
                    $this->line('name: ' . ($name ?? 'null'));
                    $this->line('label: ' . ($label ?? 'null'));
                    $firstDumped = true;
                    $this->newLine();
                }

                $success++;
            } catch (Throwable $e) {
                $failed++;
                $this->output->writeln('');
                $this->warn("Failed to derive vehicle for vevs_car_id={$vevsCarId}: " . $e->getMessage());
            } finally {
                $bar->advance();
            }
        }

        $bar->finish();
        $this->newLine(2);
        $this->info("Done. Success: {$success}, Failed: {$failed}.");

        return Command::SUCCESS;
    }
}
