<?php

declare(strict_types=1);

namespace App\Console;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Foundation\Console\Kernel as ConsoleKernel;

/** Jobs */
use App\Jobs\ChargeUpcomingBalancesJob;

/** Artisan Commands – Imports & Feeds */
use App\Console\Commands\DreamDrivesImport;
use App\Console\Commands\JimnyImport;
use App\Console\Commands\VevsSync;
use App\Console\Commands\VevsPullReservation;
use App\Console\Commands\VeVsSyncCars;
use App\Console\Commands\ImportVevsReservationCommand;

/** Artisan Commands – Bookings lifecycle & holds */
use App\Console\Commands\PlaceBondHolds;
use App\Console\Commands\Bookings\AutoCancelBookings;
use App\Console\Commands\Bookings\CloseCompletedBookings;

/** Artisan Commands – Notifications */
use App\Console\Commands\Notifications\AdminDailyDigest;
use App\Console\Commands\Notifications\CustomerPickupReminder;
use App\Console\Commands\Notifications\CustomerReturnReminder;

/** Artisan Commands – Maintenance */
use App\Console\Commands\Maintenance\PruneTempFiles;

/** Artisan Commands – Monitoring & Misc */
use App\Console\Commands\Monitoring\CheckFeeds;
use App\Console\Commands\BackfillBookingPortalTokens;
use App\Console\Commands\ShowPayment;
use App\Console\Commands\GenerateOwnerEarnings;
use App\Console\Commands\SyncJobsFromVevsBookings;

class Kernel extends ConsoleKernel
{
    /**
     * Explicitly registered custom Artisan commands.
     *
     * @var array<int, class-string>
     */
    protected $commands = [
        // Feeds / Imports
        DreamDrivesImport::class,
        JimnyImport::class,
        CheckFeeds::class,

        // VEVS
        VevsSync::class,
        VevsPullReservation::class,
        VeVsSyncCars::class,
        ImportVevsReservationCommand::class,

        // Bookings lifecycle & holds
        PlaceBondHolds::class,
        AutoCancelBookings::class,
        CloseCompletedBookings::class,

        // Notifications
        AdminDailyDigest::class,
        CustomerPickupReminder::class,
        CustomerReturnReminder::class,

        // Maintenance
        PruneTempFiles::class,

        // Misc
        BackfillBookingPortalTokens::class,
        ShowPayment::class,
        GenerateOwnerEarnings::class,
        SyncJobsFromVevsBookings::class,
    ];

    /**
     * Define the application's command schedule.
     */
    protected function schedule(Schedule $schedule): void
    {
        /** ───── Owner earnings ───── */
        $schedule->command('owners:generate-earnings')
            ->hourly()
            ->withoutOverlapping(60)
            ->onOneServer();

        /** ───── Bookings ───── */
        $schedule->job(new ChargeUpcomingBalancesJob())
            ->everyTenMinutes()
            ->withoutOverlapping(15)
            ->onOneServer();

        // VEVS syncs
        $schedule->command('vevs:sync', ['--mode' => 'weekMade'])
            ->hourly()
            ->withoutOverlapping(60)
            ->onOneServer();

        $schedule->command('vevs:sync', ['--mode' => 'weekPickup'])
            ->hourlyAt(30)
            ->withoutOverlapping(60)
            ->onOneServer();

        if (app()->environment('staging')) {
            $schedule->command('vevs:sync', ['--mode' => 'recent', '--minutes' => 15])
                ->everyTenMinutes()
                ->withoutOverlapping(15)
                ->onOneServer();
        }

        // Balance reminders
        $schedule->command('bookings:remind-balance', ['--days' => 3])
            ->dailyAt('09:00')
            ->onOneServer();

        // Bond holds
        $schedule->command('bookings:place-bond-holds')
            ->hourly()
            ->withoutOverlapping(60)
            ->onOneServer();

        // Auto-cancel stale bookings
        $schedule->command('bookings:auto-cancel')
            ->hourly()
            ->withoutOverlapping(60)
            ->onOneServer();

        // Auto-close completed bookings
        $schedule->command('bookings:close-completed')
            ->dailyAt('00:30')
            ->withoutOverlapping(60)
            ->onOneServer();

        // Daily imports
        $schedule->command('dreamdrives:import')
            ->dailyAt('09:00')
            ->withoutOverlapping(60)
            ->onOneServer();

        $schedule->command('jimny:import')
            ->dailyAt('09:00')
            ->withoutOverlapping(60)
            ->onOneServer();

        /** ───── Notifications ───── */
        $schedule->command('notify:admin-daily-digest')
            ->dailyAt('07:30')
            ->onOneServer();

        $schedule->command('notify:customer-pickup')
            ->dailyAt('12:00')
            ->onOneServer();

        $schedule->command('notify:customer-return')
            ->dailyAt('08:00')
            ->onOneServer();

        /** ───── Maintenance ───── */
        $schedule->command('queue:prune-batches', ['--hours' => 48])
            ->daily()
            ->onOneServer();

        $schedule->command('maintenance:prune-temp', ['--days' => 7])
            ->weeklyOn(1, '03:00')
            ->onOneServer();

        /** ───── Monitoring ───── */
        $schedule->command('feeds:check', ['feed' => 'dreamdrives'])
            ->dailyAt('09:05')
            ->onOneServer();

        $schedule->command('feeds:check', ['feed' => 'jimny'])
            ->dailyAt('09:05')
            ->onOneServer();

        /** ───── Holds / Automations ───── */
        $schedule->command('holds:auto-cancel')
            ->hourly()
            ->withoutOverlapping(60)
            ->onOneServer();

        $schedule->command('automations:run')
            ->everyFifteenMinutes()
            ->withoutOverlapping(30)
            ->onOneServer();
    }

    /**
     * Register the commands for the application.
     */
    protected function commands(): void
    {
        $this->load(__DIR__ . '/Commands');
        require base_path('routes/console.php');
    }

    /**
     * Use NZ time for the scheduler.
     */
    protected function scheduleTimezone(): \DateTimeZone|string|null
    {
        return 'Pacific/Auckland';
    }
}
