<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CustomerResource\Pages;
use App\Models\Customer;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Infolists\Components\Grid as InfoGrid;
use Filament\Infolists\Components\Section as InfoSection;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Route;

class CustomerResource extends Resource
{
    protected static ?string $model = Customer::class;

    protected static ?string $navigationIcon  = 'heroicon-o-user';
    protected static ?string $navigationLabel = 'Customers';
    protected static ?string $navigationGroup = 'Payments';
    protected static ?int    $navigationSort  = 3;

    public static function getModelLabel(): string
    {
        return 'Customer';
    }

    public static function getPluralModelLabel(): string
    {
        return 'Customers';
    }

    /**
     * Base query – eager load counts we care about.
     */
    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withCount('bookings');
    }

    /**
     * Create/Edit form (minimal fields for payments)
     */
    public static function form(Form $form): Form
    {
        return $form->schema([
            Forms\Components\Section::make('Customer details')
                ->columns(12)
                ->schema([
                    Forms\Components\TextInput::make('first_name')
                        ->label('First name')
                        ->required()
                        ->maxLength(100)
                        ->columnSpan(4),

                    Forms\Components\TextInput::make('last_name')
                        ->label('Last name')
                        ->required()
                        ->maxLength(100)
                        ->columnSpan(4),

                    Forms\Components\TextInput::make('company')
                        ->label('Company')
                        ->maxLength(150)
                        ->columnSpan(4),

                    Forms\Components\TextInput::make('email')
                        ->label('Email')
                        ->email()
                        ->unique(ignoreRecord: true)
                        ->required()
                        ->maxLength(191)
                        ->columnSpan(4),

                    Forms\Components\TextInput::make('phone')
                        ->label('Phone')
                        ->tel()
                        ->maxLength(50)
                        ->columnSpan(4),

                    Forms\Components\Textarea::make('notes')
                        ->label('Notes')
                        ->rows(3)
                        ->columnSpan(12),
                ]),
        ]);
    }

    /**
     * List table (lean columns, payments-focused)
     */
    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // Combined display name
                TextColumn::make('full_name')
                    ->label('Name')
                    ->sortable()
                    ->searchable(
                        // Let Filament search first/last transparently
                        query: function (Builder $query, string $search) {
                            $query->where(function (Builder $q) use ($search) {
                                $q->where('first_name', 'like', "%{$search}%")
                                  ->orWhere('last_name', 'like', "%{$search}%");
                            });
                        }
                    )
                    ->formatStateUsing(fn (Customer $record) => trim("{$record->first_name} {$record->last_name}"))
                    ->toggleable(),

                // Keep raw first/last available but hidden by default
                TextColumn::make('first_name')
                    ->label('First')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('last_name')
                    ->label('Last')
                    ->searchable()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('email')
                    ->label('Email')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('phone')
                    ->label('Phone')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('company')
                    ->label('Company')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('bookings_count')
                    ->label('Bookings')
                    ->sortable()
                    ->badge()
                    ->toggleable(),

                // Payments-related (hidden by default)
                TextColumn::make('stripe_customer_id')
                    ->label('Stripe Customer')
                    ->badge()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('default_payment_method_id')
                    ->label('Default PM')
                    ->badge()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('Created')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('Updated')
                    ->dateTime('Y-m-d H:i')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->defaultSort('created_at', 'desc')
            ->filters([
                Filter::make('has_stripe')
                    ->label('Has Stripe profile')
                    ->query(fn (Builder $query): Builder => $query->whereNotNull('stripe_customer_id')),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),

                Tables\Actions\Action::make('add_card')
                    ->label('Add/Update Card')
                    ->icon('heroicon-m-credit-card')
                    ->visible(fn () => Route::has('customers.pm.add'))
                    ->url(fn (Customer $record) => route('customers.pm.add', $record))
                    ->openUrlInNewTab(),

                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ])
            ->emptyStateHeading('No customers yet')
            ->emptyStateDescription('Create your first customer to get started.')
            ->emptyStateActions([
                Tables\Actions\CreateAction::make(),
            ]);
    }

    /**
     * View page infolist (payments-focused)
     */
    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            InfoSection::make('Customer')
                ->schema([
                    InfoGrid::make(3)->schema([
                        TextEntry::make('first_name')->label('First name'),
                        TextEntry::make('last_name')->label('Last name'),
                        TextEntry::make('company')->label('Company'),
                    ]),
                    InfoGrid::make(3)->schema([
                        TextEntry::make('email')
                            ->icon('heroicon-m-envelope')
                            ->copyable(),
                        TextEntry::make('phone')
                            ->icon('heroicon-m-phone')
                            ->copyable(),
                        TextEntry::make('created_at')
                            ->label('Created')
                            ->dateTime('Y-m-d H:i'),
                    ]),
                ]),

            InfoSection::make('Payments')
                ->collapsed()
                ->schema([
                    InfoGrid::make(2)->schema([
                        TextEntry::make('stripe_customer_id')
                            ->label('Stripe Customer ID')
                            ->placeholder('—')
                            ->copyable(),
                        TextEntry::make('default_payment_method_id')
                            ->label('Default Payment Method')
                            ->placeholder('—')
                            ->copyable(),
                    ]),
                ]),

            InfoSection::make('Notes')
                ->collapsed()
                ->schema([
                    TextEntry::make('notes')
                        ->label('Notes')
                        ->columnSpanFull(),
                ]),
        ]);
    }

    /**
     * Relations
     */
    public static function getRelations(): array
    {
        return [
            \App\Filament\Resources\CustomerResource\RelationManagers\BookingsRelationManager::class,
        ];
    }

    /**
     * Pages
     */
    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListCustomers::route('/'),
            'create' => Pages\CreateCustomer::route('/create'),
            'view'   => Pages\ViewCustomer::route('/{record}'),
            'edit'   => Pages\EditCustomer::route('/{record}/edit'),
        ];
    }

    /**
     * Global search configuration
     */
    public static function getGloballySearchableAttributes(): array
    {
        return ['first_name', 'last_name', 'email', 'phone', 'company'];
    }

    public static function getNavigationBadge(): ?string
    {
        try {
            return (string) static::getModel()::count();
        } catch (\Throwable $e) {
            return null;
        }
    }
}
