<?php

declare(strict_types=1);

namespace App\Filament\Resources;

use App\Filament\Resources\DamageResource\Pages;
use App\Models\Damage;
use App\Models\Job;
use App\Models\Vehicle;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;

class DamageResource extends Resource
{
    protected static ?string $model = Damage::class;

    protected static ?string $navigationIcon  = 'heroicon-o-exclamation-circle';
    protected static ?string $navigationGroup = 'Fleet';
    protected static ?string $navigationLabel = 'Damage';
    protected static ?int    $navigationSort  = 4;


    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Damage details')
                    ->columns(2)
                    ->schema([
                        // VEHICLE SELECT – search "name", show "name (rego)"
                        Select::make('vehicle_id')
                            ->label('Vehicle')
                            ->relationship('vehicle', 'name')
                            ->searchable()
                            ->required()
                            ->getOptionLabelFromRecordUsing(function (Vehicle $record): string {
                                $label = $record->name ?? ('Vehicle #' . $record->id);

                                if (! empty($record->registration)) {
                                    $label .= ' (' . $record->registration . ')';
                                }

                                return $label;
                            }),

                        // JOB SELECT – search & display based on external_reference
                        Select::make('job_id')
                            ->label('Job')
                            ->relationship('job', 'external_reference')   // ✅ real column
                            ->searchable()
                            ->nullable()
                            ->getOptionLabelFromRecordUsing(function (Job $record): string {
                                $parts = [];

                                // Primary FS/MG/etc code
                                if (! empty($record->external_reference)) {
                                    $parts[] = $record->external_reference;
                                }

                                // Optional internal reference if you ever use it
                                if (isset($record->reference) && ! empty($record->reference)) {
                                    $parts[] = $record->reference;
                                }

                                // Customer name for context
                                if (isset($record->customer_name) && ! empty($record->customer_name)) {
                                    $parts[] = $record->customer_name;
                                }

                                return $parts
                                    ? implode(' – ', $parts)
                                    : 'Job #' . $record->id;
                            }),

                        Select::make('reported_by_user_id')
                            ->label('Reported by')
                            ->relationship('reportedBy', 'name')
                            ->default(fn () => auth()->id())
                            ->searchable()
                            ->nullable(),

                        DateTimePicker::make('found_at')
                            ->label('Found at')
                            ->default(now())
                            ->seconds(false),
                    ]),

                Section::make('Classification')
                    ->columns(3)
                    ->schema([
                        Select::make('location')
                            ->label('Location on vehicle')
                            ->options([
                                'front_bumper'       => 'Front bumper',
                                'rear_bumper'        => 'Rear bumper',
                                'left_front_door'    => 'Left front door',
                                'right_front_door'   => 'Right front door',
                                'left_rear_door'     => 'Left rear door',
                                'right_rear_door'    => 'Right rear door',
                                'left_front_guard'   => 'Left front guard',
                                'right_front_guard'  => 'Right front guard',
                                'left_rear_quarter'  => 'Left rear quarter',
                                'right_rear_quarter' => 'Right rear quarter',
                                'roof'               => 'Roof',
                                'windscreen'         => 'Windscreen',
                                'rear_glass'         => 'Rear glass',
                                'bonnet'             => 'Bonnet',
                                'tailgate'           => 'Tailgate / boot',
                                'interior'           => 'Interior (general)',
                            ])
                            ->searchable(),

                        Select::make('type')
                            ->label('Damage type')
                            ->options([
                                'scratch'        => 'Scratch',
                                'dent'           => 'Dent',
                                'crack'          => 'Crack',
                                'chip'           => 'Chip',
                                'scuff'          => 'Scuff',
                                'paint_transfer' => 'Paint transfer',
                                'broken_part'    => 'Broken part',
                                'glass'          => 'Glass damage',
                                'interior_stain' => 'Interior stain',
                                'interior_tear'  => 'Interior tear',
                                'curb_rash'      => 'Wheel curb rash',
                                'tyre_damage'    => 'Tyre damage',
                                'missing_item'   => 'Missing accessory',
                            ])
                            ->searchable(),

                        Select::make('severity')
                            ->label('Severity')
                            ->options([
                                'minor'    => 'Minor',
                                'moderate' => 'Moderate',
                                'major'    => 'Major',
                            ]),
                    ]),

                Section::make('Repair & cost')
    ->columns(3)
    ->schema([
        TextInput::make('estimated_cost_cents')
            ->label('Estimated cost')
            ->numeric()
            ->prefix('NZ$')
            // Show dollars when editing
            ->formatStateUsing(fn ($state) => $state !== null ? number_format($state / 100, 2, '.', '') : null)
            // Convert dollars → cents when saving
            ->dehydrateStateUsing(function ($state) {
                if ($state === null || $state === '') {
                    return null;
                }

                return (int) round(((float) $state) * 100);
            })
            ->dehydrated()
            ->nullable()
            ->helperText('Enter the amount in dollars (e.g. 125.50). Stored internally as cents.'),

        Select::make('repair_status')
            ->label('Repair status')
            ->options([
                'logged'      => 'Logged',
                'quoted'      => 'Quoted',
                'approved'    => 'Approved',
                'in_progress' => 'In progress',
                'completed'   => 'Completed',
            ])
            ->default('logged'),

        Forms\Components\Toggle::make('requires_repair')
            ->label('Requires repair')
            ->default(true),

        Forms\Components\Toggle::make('is_preexisting')
            ->label('Pre-existing damage')
            ->default(false),
    ]),


                Section::make('Photos & notes')
                    ->columns(1)
                    ->schema([
                        FileUpload::make('photos')
                            ->label('Photos')
                            ->disk('public')
                            ->directory('damage-photos')
                            ->multiple()
                            ->image()
                            ->imageEditor()
                            ->openable()
                            ->downloadable()
                            ->reorderable(),

                        Textarea::make('notes')
                            ->rows(4),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                // VEHICLE column – uses vehicle.name but shows rego too
                TextColumn::make('vehicle.name')
                    ->label('Vehicle')
                    ->formatStateUsing(function (?string $state, Damage $record): string {
                        $vehicle = $record->vehicle;

                        if (! $vehicle) {
                            return '—';
                        }

                        $label = $vehicle->name ?? ('Vehicle #' . $vehicle->id);

                        if (! empty($vehicle->registration)) {
                            $label .= ' (' . $vehicle->registration . ')';
                        }

                        return $label;
                    })
                    ->searchable()
                    ->sortable(),

                // JOB column – based on external_reference so searching works
                TextColumn::make('job.external_reference')
                    ->label('Job')
                    ->formatStateUsing(function (?string $state, Damage $record): string {
                        $job = $record->job;

                        if (! $job) {
                            return '—';
                        }

                        $parts = [];

                        if (! empty($job->external_reference)) {
                            $parts[] = $job->external_reference;
                        }

                        if (isset($job->reference) && ! empty($job->reference)) {
                            $parts[] = $job->reference;
                        }

                        if (isset($job->customer_name) && ! empty($job->customer_name)) {
                            $parts[] = $job->customer_name;
                        }

                        return $parts
                            ? implode(' – ', $parts)
                            : 'Job #' . $job->id;
                    })
                    ->searchable()
                    ->sortable(),

                TextColumn::make('location')
                    ->label('Location')
                    ->badge(),

                TextColumn::make('type')
                    ->label('Type')
                    ->badge(),

                TextColumn::make('severity')
                    ->label('Severity')
                    ->badge()
                    ->formatStateUsing(
                        fn (?string $state) => $state ? ucfirst($state) : ''
                    ),

                TextColumn::make('repair_status')
                    ->label('Repair')
                    ->badge(),

                TextColumn::make('found_at')
                    ->dateTime()
                    ->label('Found at')
                    ->sortable(),
            ])
            ->defaultSort('found_at', 'desc')
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListDamages::route('/'),
            'create' => Pages\CreateDamage::route('/create'),
            'view'   => Pages\ViewDamage::route('/{record}'),
            'edit'   => Pages\EditDamage::route('/{record}/edit'),
        ];
    }
}
