<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DepositResource\Pages;
use App\Filament\Resources\PaymentResource;
use App\Models\Payment;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Actions\Action;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

// app/Filament/Resources/DepositResource.php  (merged with minimal changes)

class DepositResource extends Resource
{
    protected static ?string $model = Payment::class;

    /** URLs & nav copy now reflect Bond Holds */
    protected static ?string $slug = 'bond-holds';
    protected static ?string $navigationIcon   = 'heroicon-o-banknotes';
    protected static ?string $navigationGroup  = 'Payments';
    protected static ?string $navigationLabel  = 'Bond Holds';
    protected static ?string $pluralModelLabel = 'Bond Holds';
    protected static ?int    $navigationSort   = 11;

    public static function getNavigationBadge(): ?string
    {
        return (string) static::holdsQuery(Payment::query())->count();
    }

    /**
     * "Hold" = bond_hold + hold-ish statuses (no deposit_confirmed_at gate here)
     */
    protected static function holdsQuery(Builder $q): Builder
    {
        // Only bond holds
        $q->where('type', 'bond_hold');

        // Status looks like a hold
        $q->where(function (Builder $i) {
            $i->whereIn('status', ['pending', 'authorized', 'requires_capture']);

            // Stripe PI status
            if (Schema::hasColumn('payments', 'stripe_payment_intent_status')) {
                $i->orWhere('stripe_payment_intent_status', 'requires_capture');
            }

            // Manual-capture hint
            if (Schema::hasColumn('payments', 'capture_method')) {
                $i->orWhere('capture_method', 'manual');
            }

            // Explicit hold flag
            if (Schema::hasColumn('payments', 'is_hold')) {
                $i->orWhere('is_hold', true);
            }
        });

        // Card-only (if present)
        if (Schema::hasColumn('payments', 'mechanism')) {
            $q->whereIn('mechanism', ['card', 'record_only']);
        }

        return $q;
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(fn (Builder $query) => static::holdsQuery($query))
            ->defaultSort('created_at', 'desc')

            // OPEN the underlying Payment view instead of a DepositResource page
            ->recordUrl(fn (Payment $record) => PaymentResource::getUrl('view', ['record' => $record]))
            ->recordAction(null)

            ->columns([
                Tables\Columns\TextColumn::make('job.external_reference')->label('Job')->sortable()->searchable(),
                Tables\Columns\TextColumn::make('amount_cents')
                    ->label('Amount')->sortable()
                    ->formatStateUsing(fn ($state, Payment $record) =>
                        ($record->currency ?? 'NZD') . ' ' . number_format(((int) $state) / 100, 2)
                    ),
                Tables\Columns\TextColumn::make('status')
                    ->label('Status')->badge()
                    ->colors([
                        'warning' => ['pending', 'requires_capture'],
                        'info'    => ['authorized'],
                        'success' => ['succeeded', 'captured'],
                        'gray'    => ['canceled'],
                        'danger'  => ['failed'],
                    ])
                    ->sortable(),
                Tables\Columns\TextColumn::make('stripe_payment_intent_status')
                    ->label('PI status')->toggleable(isToggledHiddenByDefault: true),
                Tables\Columns\TextColumn::make('created_at')->label('Created')->dateTime()->since()->sortable(),
            ])
            ->headerActions([
                Action::make('newBondHold')
                    ->label('New bond hold')
                    ->icon('heroicon-o-plus')
                    ->button()
                    ->url(
                        PaymentResource::getUrl('create', [
                            'type' => 'bond_hold',
                        ])
                    ),
            ])
            ->actions([
                // Removed View/Edit actions to avoid hitting Deposit pages
                Tables\Actions\DeleteAction::make(),

                // Open underlying payment (also available on row click)
                Action::make('openPayment')
                    ->label('Open payment')
                    ->icon('heroicon-o-arrow-top-right-on-square')
                    ->url(fn (Payment $record) => PaymentResource::getUrl('view', ['record' => $record]))
                    ->openUrlInNewTab(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            // Keep only the index so /bond-holds/{id} isn't routed at all
            'index' => Pages\ListDeposits::route('/'),
        'view'  => Pages\ViewDeposit::route('/{record}'),
        ];
    }
}
