<?php

namespace App\Filament\Resources\DepositResource\Pages;

use App\Filament\Resources\DepositResource;
use App\Filament\Resources\PaymentResource;
use App\Models\Payment;
use Filament\Actions\Action;
use Filament\Infolists\Components\Grid;
use Filament\Infolists\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;

class ViewDeposit extends ViewRecord
{
    public static string $resource = DepositResource::class;

    public function getTitle(): string
    {
        return 'View Bond Hold'; // was: View Deposit
    }

    /** Infolist: deposit / hold details */
    public function infolist(Infolist $infolist): Infolist
    {
        /** @var Payment $p */
        $p = $this->record;

        $amount   = number_format(((int) $p->amount_cents) / 100, 2);
        $currency = $p->currency ?? 'NZD';

        return $infolist->schema([
            Section::make('Deposit')
                ->schema([
                    Grid::make(2)->schema([
                        TextEntry::make('job.external_reference')
                            ->label('Job Reference')
                            ->default($p->job->external_reference ?? '—'),

                        TextEntry::make('status')
                            ->badge()
                            ->color(fn () => match ($p->status) {
                                'pending', 'requires_capture' => 'warning',
                                'authorized'                  => 'info',
                                'succeeded', 'captured'       => 'success',
                                'canceled'                    => 'gray',
                                'failed'                      => 'danger',
                                default                       => 'gray',
                            }),

                        TextEntry::make('public_link')
                            ->label('Public link')
                            ->default(fn () => $this->payUrl() ?? '—')
                            ->url(fn () => $this->payUrl() ?: null, shouldOpenInNewTab: true)
                            ->copyable()
                            ->hidden(fn () => ! $this->payUrl()),

                        TextEntry::make('amount')
                            ->label('Amount')
                            ->default($currency . ' ' . $amount),

                        TextEntry::make('mechanism')
                            ->label('Mechanism')
                            ->default($p->mechanism ?? 'card'),

                        TextEntry::make('provider')
                            ->label('Provider')
                            ->default($p->provider ?? 'stripe'),

                        TextEntry::make('capture_method')
                            ->label('Capture method')
                            ->default($p->capture_method ?? '—')
                            ->hidden(fn () => ! $this->hasColumn('capture_method')),

                        TextEntry::make('stripe_payment_intent_status')
                            ->label('PI status')
                            ->default($p->stripe_payment_intent_status ?? '—')
                            ->hidden(fn () => ! $this->hasColumn('stripe_payment_intent_status')),

                        TextEntry::make('deposit_confirmed_at')
                            ->label('Deposit confirmed at')
                            ->dateTime()
                            ->default($this->record->deposit_confirmed_at?->toDateTimeString() ?? '—'),

                        TextEntry::make('created_at')->dateTime()->label('Created'),
                        TextEntry::make('updated_at')->dateTime()->label('Updated'),

                        TextEntry::make('link.public_token')
                            ->label('Public token')
                            ->default($p->link->public_token ?? '—')
                            ->copyable()
                            ->hidden(fn () => empty($p->link?->public_token)),

                        TextEntry::make('expires_at')
                            ->label('Link expires')
                            ->default(optional($p->link?->expires_at)->toDateTimeString() ?? '—')
                            ->hidden(fn () => empty($p->link?->expires_at)),
                    ]),
                ]),
        ]);
    }

    /** Header actions */
    protected function getHeaderActions(): array
    {
        /** @var Payment $p */
        $p = $this->record;

        return [
            // ✅ Mark deposit as paid (uses the same model method)
            Action::make('markPaid')
                ->label('Mark as paid')
                ->icon('heroicon-o-check-badge')
                ->color('primary')
                ->requiresConfirmation()
                ->visible(fn (Payment $record) => $record->isDeposit() && ! $record->isDepositConfirmed())
                ->action(function (Payment $record) {
                    try {
                        // call your model method; pass actor if your signature supports it
                        $record->markDepositAsPaid(auth()->id() ?: null);
                        Notification::make()->title('Deposit marked as paid')->success()->send();
                        $this->record->refresh();
                    } catch (\Throwable $e) {
                        Notification::make()
                            ->title('Couldn’t mark as paid')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            // Capture hold
            Action::make('capture')
                ->label('Capture')
                ->icon('heroicon-o-lock-closed')
                ->color('success')
                ->requiresConfirmation()
                ->visible(fn () => $this->isHold($p))
                ->action(function () use ($p) {
                    try {
                        if (method_exists($p, 'captureHold')) {
                            $p->captureHold();
                            Notification::make()->title('Deposit captured')->success()->send();
                        } elseif (method_exists($p, 'capture')) {
                            $p->capture();
                            Notification::make()->title('Deposit captured')->success()->send();
                        } else {
                            Notification::make()
                                ->title('Capture not wired')
                                ->body('Add Payment::captureHold() or Payment::capture() to enable this.')
                                ->warning()
                                ->send();
                        }
                        $this->record->refresh();
                    } catch (\Throwable $e) {
                        Log::error('Capture failed', ['err' => $e->getMessage()]);
                        Notification::make()
                            ->title('Capture failed')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            // Release hold
            Action::make('release')
                ->label('Release')
                ->icon('heroicon-o-lock-open')
                ->color('danger')
                ->requiresConfirmation()
                ->visible(fn () => $this->isHold($p))
                ->action(function () use ($p) {
                    try {
                        if (method_exists($p, 'releaseHold')) {
                            $p->releaseHold();
                            Notification::make()->title('Deposit released')->success()->send();
                        } elseif (method_exists($p, 'cancel')) {
                            $p->cancel();
                            Notification::make()->title('Deposit released')->success()->send();
                        } else {
                            Notification::make()
                                ->title('Release not wired')
                                ->body('Add Payment::releaseHold() or Payment::cancel() to enable this.')
                                ->warning()
                                ->send();
                        }
                        $this->record->refresh();
                    } catch (\Throwable $e) {
                        Log::error('Release failed', ['err' => $e->getMessage()]);
                        Notification::make()
                            ->title('Release failed')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            // Open public pay page
            Action::make('openPayPage')
                ->label('Open pay page')
                ->icon('heroicon-o-arrow-top-right-on-square')
                ->url(fn () => $this->payUrl())
                ->openUrlInNewTab()
                ->visible(fn () => (bool) $this->payUrl()),

            // Open internal payment record
            Action::make('openPayment')
                ->label('Open payment')
                ->icon('heroicon-o-credit-card')
                ->url(fn () => PaymentResource::getUrl('view', ['record' => $this->record]))
                ->openUrlInNewTab(),
        ];
    }

    /** Helpers */
    protected function isHold(Payment $p): bool
    {
        if (in_array($p->status, ['pending', 'authorized', 'requires_capture'], true)) {
            return true;
        }
        if ($this->hasColumn('stripe_payment_intent_status') && $p->stripe_payment_intent_status === 'requires_capture') {
            return true;
        }
        if ($this->hasColumn('capture_method') && $p->capture_method === 'manual' && ! in_array($p->status, ['succeeded', 'captured'], true)) {
            return true;
        }
        if ($this->hasColumn('is_hold') && (bool) $p->is_hold) {
            return true;
        }
        return false;
    }

    protected function payUrl(): ?string
    {
        /** @var Payment $p */
        $p = $this->record;
        $token = $p->link->public_token ?? null;
        if (! $token) {
            return null;
        }

        foreach (['pay.show', 'pay.page', 'public.pay.show'] as $candidate) {
            if (app('router')->has($candidate)) {
                return route($candidate, ['token' => $token]);
            }
        }

        return url('/pay/' . $token);
    }

    protected function hasColumn(string $name): bool
    {
        static $columns;
        if ($columns === null) {
            $columns = collect(Schema::getColumnListing('payments'))->flip();
        }
        return $columns->has($name);
    }
}
