<?php
// app/Filament/Resources/JobResource/Pages/EditJob.php

declare(strict_types=1);

namespace App\Filament\Resources\JobResource\Pages;

use App\Enums\OwnerLedgerEntryType;
use App\Filament\Resources\JobResource;
use App\Filament\Resources\VehicleResource;
use App\Mail\PaymentRequestMail;
use App\Models\Job;
use App\Models\OwnerLedgerEntry;
use App\Models\Payment;
use App\Services\ExternalSyncService;
use App\Support\Payments\OneOffPaymentService;
use Filament\Actions;
use Filament\Forms;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Throwable;

class EditJob extends EditRecord
{
    protected static string $resource = JobResource::class;

    // -------------------------------------------------------------------------
    // Layout
    // -------------------------------------------------------------------------

    public function getMaxContentWidth(): ?string
    {
        return 'full';
    }

    public function hasFullWidthForm(): bool
    {
        return true;
    }

    // -------------------------------------------------------------------------
    // Form normalisation
    // -------------------------------------------------------------------------

    /**
     * Normalise form data BEFORE saving the Job.
     *
     * Currently we let the form components handle cents/dollars conversion,
     * so this just returns the data unchanged. Kept for future tweaks.
     */
    protected function mutateFormDataBeforeSave(array $data): array
    {
        // Dates are already in app timezone; nothing special to do here.
        return $data;
    }

    // -------------------------------------------------------------------------
    // After save hook
    // -------------------------------------------------------------------------

    /**
     * After save:
     * - Optionally create a booking_deposit Payment based on the Deposit tab.
     * - Create / update extra mileage owner ledger entry with 100% to owner.
     */
    protected function afterSave(): void
    {
        /** @var Job $job */
        $job = $this->record;

        // Use RAW state so we can see fields with dehydrated(false)
        $state = $this->form->getRawState();

        // -------------------------------------------------------------
        // 1) Deposit from "Deposit" tab (existing behaviour)
        // -------------------------------------------------------------
        $depositRaw = $state['deposit_amount'] ?? null;

        $depositCents = null;
        if ($depositRaw !== null && $depositRaw !== '') {
            $normalised   = preg_replace('/[^\d.]/', '', (string) $depositRaw);
            $depositCents = (int) round(((float) $normalised) * 100);
        }

        if ($depositCents && $depositCents > 0) {
            Payment::create([
                'job_id'       => $job->id,
                'type'         => 'booking_deposit',
                'amount_cents' => $depositCents,
                'currency'     => $job->currency ?? 'NZD',
                'status'       => 'succeeded',
                'mechanism'    => 'manual',
                'metadata'     => [
                    'note' => 'Created from EditJob form (Deposit tab).',
                ],
            ]);

            Notification::make()
                ->title('Deposit payment created')
                ->body('A booking_deposit payment has been created for this job.')
                ->success()
                ->send();
        }

        // -------------------------------------------------------------
        // 2) Extra mileage – owner gets 100% of this charge
        // -------------------------------------------------------------
        // NOTE: adjust the key below to match your form field
        // (e.g. 'extra_mileage_amount' / 'extra_mileage_charge' etc.)
        $extraMileageRaw = $state['extra_mileage_amount'] ?? null;

        $extraMileageCents = null;
        if ($extraMileageRaw !== null && $extraMileageRaw !== '') {
            $normalised        = preg_replace('/[^\d.]/', '', (string) $extraMileageRaw);
            $extraMileageCents = (int) round(((float) $normalised) * 100);
        }

        if ($extraMileageCents && $extraMileageCents > 0) {
            $vehicle = $job->vehicle;

            // Only record if this job has a consignment owner
            if ($vehicle && $vehicle->owner_id) {
                // Optional: persist the extra mileage on the Job itself
                // if you have an extra_mileage_cents column:
                // $job->extra_mileage_cents = $extraMileageCents;
                // $job->save();

                // Use updateOrCreate so edits stay in sync
                OwnerLedgerEntry::updateOrCreate(
                    [
                        'owner_id'   => $vehicle->owner_id,
                        'vehicle_id' => $vehicle->id,
                        'job_id'     => $job->id,
                        'type'       => OwnerLedgerEntryType::Earning,
                        'description'=> 'Extra mileage charge',
                    ],
                    [
                        'amount_cents'      => $extraMileageCents,
                        'currency_code'     => $job->currency ?? 'NZD',
                        'occurred_at'       => now(),
                        // If you have owner_share_cents + effective_owner_share_cents,
                        // this ensures 100% of the charge goes to the owner:
                        'owner_share_cents' => $extraMileageCents,
                    ],
                );
            }
        }
    }

    // -------------------------------------------------------------------------
    // Header actions
    // -------------------------------------------------------------------------

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('sendPaymentRequest')
                ->label('Send Payment Request')
                ->icon('heroicon-o-envelope')
                ->requiresConfirmation()
                ->visible(fn (Job $record) => (bool) $record->customer_email)
                ->action(function (Job $record, array $data): void {
                    $this->handleSendPaymentRequest($record, $data);
                })
                ->form([
                    Forms\Components\TextInput::make('amount')
                        ->label('Amount (NZD)')
                        ->prefix('$')
                        ->numeric()
                        ->required()
                        ->default(fn (Job $record) => $record->charge_amount_cents
                            ? number_format($record->charge_amount_cents / 100, 2, '.', '')
                            : null),
                    Forms\Components\TextInput::make('email')
                        ->label('Email')
                        ->email()
                        ->required()
                        ->default(fn (Job $record) => $record->customer_email),
                    Forms\Components\Textarea::make('message')
                        ->label('Message (optional)')
                        ->rows(3)
                        ->placeholder('Include any special instructions or notes for the customer.'),
                ]),

            Actions\Action::make('syncWithExternal')
                ->label('Sync with external system')
                ->icon('heroicon-o-arrow-path')
                ->requiresConfirmation()
                ->action(function (Job $record): void {
                    $this->handleExternalSync($record);
                }),

            Actions\Action::make('vehicleMaintenance')
                ->label('Vehicle maintenance')
                ->icon('heroicon-o-wrench')
                ->visible(fn (Job $record): bool => ! is_null($record->vehicle_id))
                ->url(fn (Job $record): ?string => $record->vehicle_id
                    ? VehicleResource::getUrl('edit', ['record' => $record->vehicle_id])
                    : null)
                ->openUrlInNewTab(),

            Actions\DeleteAction::make(),
        ];
    }

    // -------------------------------------------------------------------------
    // Payment request handler
    // -------------------------------------------------------------------------

    protected function handleSendPaymentRequest(Job $job, array $data): void
    {
        $amountStr   = (string) ($data['amount'] ?? '0');
        $amountNorm  = preg_replace('/[^\d.]/', '', $amountStr);
        $amountCents = (int) round(((float) $amountNorm) * 100);
        $email       = (string) ($data['email'] ?? $job->customer_email);
        $message     = (string) ($data['message'] ?? '');

        if ($amountCents <= 0) {
            Notification::make()
                ->title('Invalid amount')
                ->body('Please provide a valid amount greater than zero.')
                ->danger()
                ->send();

            return;
        }

        if (! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            Notification::make()
                ->title('Invalid email')
                ->body('Please provide a valid email address.')
                ->danger()
                ->send();

            return;
        }

        try {
            DB::beginTransaction();

            $payment = Payment::create([
                'job_id'       => $job->id,
                'type'         => 'one_off',
                'mechanism'    => 'card',
                'amount_cents' => $amountCents,
                'currency'     => $job->currency ?? 'NZD',
                'status'       => 'pending',
                'metadata'     => [
                    'note' => 'Payment request sent from EditJob page.',
                ],
            ]);

            /** @var OneOffPaymentService $service */
            $service = app(OneOffPaymentService::class);
            $payLink = $service->createPaymentLink($payment);

            $payment->public_pay_url = $payLink;
            $payment->save();

            Mail::to($email)->send(
                new PaymentRequestMail(
                    job: $job,
                    payment: $payment,
                    message: $message,
                )
            );

            DB::commit();

            Notification::make()
                ->title('Payment request sent')
                ->body("A payment request email has been sent to {$email}.")
                ->success()
                ->send();
        } catch (Throwable $e) {
            DB::rollBack();
            report($e);

            Notification::make()
                ->title('Payment request failed')
                ->body('An error occurred while sending the payment request. Please check logs.')
                ->danger()
                ->send();
        }
    }

    // -------------------------------------------------------------------------
    // External sync handler
    // -------------------------------------------------------------------------

    protected function handleExternalSync(Job $job): void
    {
        /** @var ExternalSyncService $syncService */
        $syncService = app(ExternalSyncService::class);

        try {
            $syncService->syncJob($job);

            $this->record->refresh();

            Notification::make()
                ->title('External sync completed')
                ->body('The VEVS pricing snapshot and related fields have been refreshed from the external booking.')
                ->success()
                ->send();
        } catch (Throwable $e) {
            Log::error('[Filament] External sync failed', [
                'job_id'    => $job->id,
                'reference' => $job->booking_reference ?? null,
                'message'   => $e->getMessage(),
                'trace'     => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->title('External sync failed')
                ->body($e->getMessage())
                ->danger()
                ->persistent()
                ->send();
        }
    }

    // -------------------------------------------------------------------------
    // Record update override
    // -------------------------------------------------------------------------

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        $record->update($data);

        return $record->fresh();
    }
}
