<?php
// ============================================================================
// File: app/Filament/Resources/JobResource/Pages/ViewJob.php
// - View page structured to mirror Edit tabs (Summary, Customer, Dates,
//   Financials, Hold)
// - Hold buttons are ALWAYS visible (disabled with tooltip if no URL yet)
// - Clear hold status (Not created / Needs authorisation / Authorised —
//   not captured / Pending / Captured / No hold)
// - Bump reminder when reservation date is approaching & hold isn't ready
// - Route-safe URL building, no references to non-existent DB columns
// - Keeps your payment request actions & utilities
// - Adds Sync from VEVS button and shows external reference in Summary
// ============================================================================

declare(strict_types=1);

namespace App\Filament\Resources\JobResource\Pages;

use App\Filament\Resources\DepositResource;
use App\Filament\Resources\JobResource;
use App\Filament\Resources\JobResource\Widgets\HoldReleaseNotice;
use App\Http\Controllers\JobController;
use App\Jobs\SendPaymentRequestEmail;
use App\Models\Deposit;
use App\Models\Job;
use App\Services\ExternalSyncService;
use Carbon\Carbon;
use Filament\Actions;
use Filament\Forms;
use Filament\Forms\Components\DateTimePicker;
use Filament\Infolists\Components\Section as InfoSection;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ViewRecord;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\URL;

class ViewJob extends ViewRecord
{
    protected static string $resource = JobResource::class;

    /** How many days before start to warn if hold not ready. */
    protected int $holdReminderDays = 5;

    /** Strip heavy fields before fill (defensive). */
    protected function mutateFormDataBeforeFill(array $data): array
    {
        unset($data['meta'], $data['comms_log'], $data['billing_address']);

        return $data;
    }

    /** Banner under header actions. */
    protected function getHeaderWidgets(): array
    {
        return [HoldReleaseNotice::class];
    }

    /** Header actions. */
    protected function getHeaderActions(): array
    {
        // Fire a non-fatal bump if start is near and hold needs action.
        $this->maybeNotifyHoldReminder();

        $missingEmailTooltip = fn () =>
            blank($this->getRecord()?->customer_email) ? 'Add a customer email first' : null;

        $missingStartOrEmailTooltip = fn () =>
            blank($this->getRecord()?->start_at)
                ? 'Set a start date/time first'
                : (blank($this->getRecord()?->customer_email) ? 'Add a customer email first' : null);

        $holdUrl      = fn () => $this->holdPageUrl($this->getRecord());
        $holdDisabled = fn () => ! (bool) $holdUrl();

        return [
            // --- Send now
            Actions\Action::make('emailPaymentRequest')
                ->label('Email payment request')
                ->icon('heroicon-o-paper-airplane')
                ->color('primary')
                ->button()
                ->requiresConfirmation()
                ->disabled(fn () => blank($this->getRecord()?->customer_email))
                ->tooltip($missingEmailTooltip)
                ->action('sendPaymentRequest'),

            // --- Sync from VEVS
            Actions\Action::make('syncFromVevs')
                ->label('Sync from VEVS')
                ->icon('heroicon-o-arrow-path')
                ->color('secondary')
                ->requiresConfirmation()
                ->visible(fn (Job $record) => filled($record->booking_reference))
                ->action(function (ExternalSyncService $syncService): void {
                    /** @var Job $job */
                    $job = $this->getRecord();

                    try {
                        $syncService->syncJob($job);

                        Notification::make()
                            ->title('Job synced from VEVS')
                            ->body('The job details have been refreshed from the external reservation.')
                            ->success()
                            ->send();

                        $this->refreshRecord();
                    } catch (\Throwable $e) {
                        Log::error('vevs.sync.failed', [
                            'job_id' => $job->getKey(),
                            'err'    => $e->getMessage(),
                        ]);

                        Notification::make()
                            ->title('Sync from VEVS failed')
                            ->body('Check logs for details: ' . $e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            // --- Schedule variants
            Actions\ActionGroup::make([
                Actions\Action::make('schedule2DaysBefore')
                    ->label('2 days before hire')
                    ->icon('heroicon-o-clock')
                    ->disabled(fn () => blank($this->getRecord()?->start_at) || blank($this->getRecord()?->customer_email))
                    ->tooltip($missingStartOrEmailTooltip)
                    ->action(function () {
                        /** @var Job $job */
                        $job = $this->getRecord();
                        if (! $job->start_at) {
                            Notification::make()
                                ->title('Cannot schedule')
                                ->body('This job has no start date/time.')
                                ->danger()
                                ->send();

                            return;
                        }

                        $tz   = config('app.timezone');
                        $when = $job->start_at->copy()
                            ->timezone($tz)
                            ->setTime(9, 0)
                            ->subDays(2);

                        $note = null;
                        if ($when->isPast()) {
                            $when = now()->addMinute();
                            $note = 'Start-minus-2-days is in the past; sending ASAP.';
                        }

                        $job->forceFill([
                            'payment_request_scheduled_at' => $when->copy()->utc(),
                        ])->save();

                        SendPaymentRequestEmail::dispatch($job->id)->delay($when);

                        Notification::make()
                            ->title('Payment request scheduled')
                            ->body(($note ? $note . ' ' : '') . 'Will send at ' . $when->timezone($tz)->toDayDateTimeString() . '.')
                            ->success()
                            ->send();
                    }),

                Actions\Action::make('schedule5DaysBefore')
                    ->label('5 days before hire')
                    ->icon('heroicon-o-clock')
                    ->disabled(fn () => blank($this->getRecord()?->start_at) || blank($this->getRecord()?->customer_email))
                    ->tooltip($missingStartOrEmailTooltip)
                    ->action(function () {
                        /** @var Job $job */
                        $job = $this->getRecord();
                        if (! $job->start_at) {
                            Notification::make()
                                ->title('Cannot schedule')
                                ->body('This job has no start date/time.')
                                ->danger()
                                ->send();

                            return;
                        }

                        $tz   = config('app.timezone');
                        $when = $job->start_at->copy()
                            ->timezone($tz)
                            ->setTime(9, 0)
                            ->subDays(5);

                        $note = null;
                        if ($when->isPast()) {
                            $when = now()->addMinute();
                            $note = 'Start-minus-5-days is in the past; sending ASAP.';
                        }

                        $job->forceFill([
                            'payment_request_scheduled_at' => $when->copy()->utc(),
                        ])->save();

                        SendPaymentRequestEmail::dispatch($job->id)->delay($when);

                        Notification::make()
                            ->title('Payment request scheduled')
                            ->body(($note ? $note . ' ' : '') . 'Will send at ' . $when->timezone($tz)->toDayDateTimeString() . '.')
                            ->success()
                            ->send();
                    }),

                Actions\Action::make('schedulePaymentRequest')
                    ->label('Pick date & time…')
                    ->icon('heroicon-o-calendar-days')
                    ->form([
                        DateTimePicker::make('when')
                            ->label('Send at')
                            ->seconds(false)
                            ->required()
                            ->minDate(now())
                            ->default(now()->addMinutes(15)),
                        Forms\Components\TextInput::make('note')
                            ->label('Optional note')
                            ->helperText('Shown in the email body (optional).'),
                    ])
                    ->disabled(fn () => blank($this->getRecord()?->customer_email))
                    ->tooltip($missingEmailTooltip)
                    ->action(function (array $data) {
                        /** @var Job $job */
                        $job  = $this->getRecord();
                        $when = Carbon::parse($data['when'], config('app.timezone'))->utc();

                        $job->forceFill([
                            'payment_request_scheduled_at' => $when,
                        ])->save();

                        SendPaymentRequestEmail::dispatch($job->id, [
                            'note' => $data['note'] ?? null,
                        ])->delay($when);

                        Notification::make()
                            ->title('Payment request scheduled')
                            ->body($when->timezone(config('app.timezone'))->toDayDateTimeString())
                            ->success()
                            ->send();
                    }),
            ])
                ->label('Schedule send')
                ->icon('heroicon-m-clock')
                ->color('gray')
                ->button(),

            // --- Utilities
            Actions\ActionGroup::make([
                Actions\Action::make('viewDeposit')
                    ->label('View Bond Hold')
                    ->icon('heroicon-o-banknotes')
                    ->url(function () {
                        /** @var Job $job */
                        $job     = $this->getRecord();
                        $deposit = $this->latestDepositForJob($job);

                        return $deposit?->getKey()
                            ? DepositResource::getUrl('view', ['record' => $deposit->getKey()])
                            : null;
                    })
                    ->visible(fn () => (bool) $this->latestDepositForJob($this->getRecord()))
                    ->openUrlInNewTab(),

                Actions\Action::make('copyPayLink')
                    ->label('Copy pay link')
                    ->icon('heroicon-o-clipboard')
                    ->modalHeading('Copy pay link')
                    ->modalSubmitActionLabel('Close')
                    ->form([
                        Forms\Components\TextInput::make('pay_link')
                            ->formatStateUsing(fn () => $this->signedPayUrl($this->getRecord()))
                            ->readOnly()
                            ->columnSpanFull()
                            ->extraAttributes([
                                'x-data' => '{ copied: false }',
                                'x-ref'  => 'pay',
                                '@focus' => '$refs.pay.select()',
                            ])
                            ->suffixActions([
                                Forms\Components\Actions\Action::make('copy')
                                    ->icon('heroicon-o-clipboard-document')
                                    ->label('Copy')
                                    ->extraAttributes([
                                        'type' => 'button',
                                        'x-on:click' =>
                                            "navigator.clipboard.writeText(\$refs.pay.value).then(() => {
                                                window.dispatchEvent(new CustomEvent('filament-notify', {
                                                    detail: { status: 'success', message: 'Pay link copied' }
                                                }));
                                            }).catch(() => {
                                                window.dispatchEvent(new CustomEvent('filament-notify', {
                                                    detail: { status: 'danger', message: 'Copy failed' }
                                                }));
                                            });",
                                    ]),
                            ])
                            ->helperText('Click Copy, or select and press ⌘C / CTRL+C'),
                    ]),

                // --- Copy hold link (ALWAYS visible; disabled if URL unavailable)
                Actions\Action::make('copyHoldLink')
                    ->label('Copy hold link')
                    ->icon('heroicon-o-clipboard')
                    ->modalHeading('Copy hold link')
                    ->modalSubmitActionLabel('Close')
                    ->visible(fn () => true)
                    ->disabled(fn () => $holdDisabled())
                    ->tooltip(fn () => $holdDisabled() ? $this->holdDisabledTooltip() : null)
                    ->form([
                        Forms\Components\TextInput::make('hold_link')
                            ->formatStateUsing(function () {
                                $url = $this->holdPageUrl($this->getRecord());

                                return $url ?: 'Hold page not available yet';
                            })
                            ->readOnly()
                            ->columnSpanFull()
                            ->extraAttributes([
                                'x-data' => '{ copied: false }',
                                'x-ref'  => 'hold',
                                '@focus' => '$refs.hold.select()',
                            ])
                            ->suffixActions([
                                Forms\Components\Actions\Action::make('copy')
                                    ->icon('heroicon-o-clipboard-document')
                                    ->label('Copy')
                                    ->disabled(fn () => $holdDisabled())
                                    ->extraAttributes([
                                        'type' => 'button',
                                        'x-on:click' =>
                                            "navigator.clipboard.writeText(\$refs.hold.value).then(() => {
                                                window.dispatchEvent(new CustomEvent('filament-notify', {
                                                    detail: { status: 'success', message: 'Hold link copied' }
                                                }));
                                            }).catch(() => {
                                                window.dispatchEvent(new CustomEvent('filament-notify', {
                                                    detail: { status: 'danger', message: 'Copy failed' }
                                                }));
                                            });",
                                    ]),
                            ])
                            ->helperText('Click Copy, or select and press ⌘C / CTRL+C'),
                    ]),

                // --- Open hold page (ALWAYS visible; disabled if URL unavailable)
                Actions\Action::make('openHoldPage')
                    ->label('Open hold page')
                    ->icon('heroicon-o-link')
                    ->visible(fn () => true)
                    ->disabled(fn () => $holdDisabled())
                    ->tooltip(fn () => $holdDisabled() ? $this->holdDisabledTooltip() : null)
                    ->url(fn () => $holdUrl())
                    ->openUrlInNewTab(),

                Actions\Action::make('openPayPage')
                    ->label('Open pay page')
                    ->icon('heroicon-o-link')
                    ->url(fn () => $this->signedPayUrl($this->getRecord()))
                    ->openUrlInNewTab(),

                Actions\Action::make('editJob')
                    ->label('Edit job')
                    ->icon('heroicon-o-pencil-square')
                    ->url(fn () => static::getResource()::getUrl('edit', ['record' => $this->getRecord()]))
                    ->openUrlInNewTab(false),
            ])
                ->label('More')
                ->icon('heroicon-m-ellipsis-vertical')
                ->color('gray')
                ->button(),
        ];
    }

    // =========================================================================
    // Read-only view layout
    // =========================================================================

    /** Infolist structured similarly to the Edit tabs. */
    public function infolist(Infolist $infolist): Infolist
    {
        return $infolist->schema([
            // SUMMARY — high level booking info
            InfoSection::make('Summary')
                ->schema([
                    TextEntry::make('external_reference')
                        ->label('Job reference')
                        ->copyable(),

                    TextEntry::make('booking_reference')
                        ->label('External reference')
                        ->copyable()
                        ->visible(fn (Job $record) => filled($record->booking_reference)),

                    TextEntry::make('flow.name')
                        ->label('Flow'),

                    TextEntry::make('status')
                        ->label('Status'),

                    TextEntry::make('vehicle.name')
                        ->label('Vehicle'),

                    TextEntry::make('source_system')
                        ->label('Source system'),

                    TextEntry::make('customer_name')
                        ->label('Customer'),

                    TextEntry::make('customer_email')
                        ->label('Email'),

                    TextEntry::make('customer_phone')
                        ->label('Phone'),

                    TextEntry::make('start_at')
                        ->label('Start at')
                        ->dateTime('D d M, H:i'),

                    TextEntry::make('end_at')
                        ->label('End at')
                        ->dateTime('D d M, H:i'),
                ])
                ->columns(3),

            // CUSTOMER — mirrors Customer tab on Edit
            InfoSection::make('Customer')
                ->schema([
                    TextEntry::make('customer_name')->label('Name'),
                    TextEntry::make('customer_email')->label('Email'),
                    TextEntry::make('customer_phone')->label('Phone'),
                    TextEntry::make('customer_notes')->label('Notes'),
                ])
                ->columns(2),

            // DATES — matches Dates tab on Edit
            InfoSection::make('Dates')
                ->schema([
                    TextEntry::make('start_at')
                        ->label('Start at')
                        ->dateTime('D d M Y, H:i'),

                    TextEntry::make('end_at')
                        ->label('End at')
                        ->dateTime('D d M Y, H:i'),

                    TextEntry::make('created_at')
                        ->label('Created at')
                        ->dateTime('D d M Y, H:i'),

                    TextEntry::make('updated_at')
                        ->label('Last updated')
                        ->dateTime('D d M Y, H:i'),
                ])
                ->columns(4),

            // FINANCIALS — matches Financials tab on Edit
            InfoSection::make('Financials')
                ->schema([
                    TextEntry::make('charge_amount_cents')
                        ->label('Charge (NZD)')
                        ->formatStateUsing(
                            fn ($state) => 'NZ$ ' . number_format(((int) ($state ?? 0)) / 100, 2)
                        ),

                    TextEntry::make('hold_from_flow')
                        ->label('Hold (NZD) — from Flow')
                        ->getStateUsing(
                            fn (Job $record) => (int) (optional($record->flow)->hold_amount_cents ?? 0)
                        )
                        ->formatStateUsing(
                            fn ($state) => 'NZ$ ' . number_format(((int) $state) / 100, 2)
                        ),

                    TextEntry::make('paid_amount_cents')
                        ->label('Paid to date')
                        ->formatStateUsing(
                            fn ($state) => 'NZ$ ' . number_format(((int) ($state ?? 0)) / 100, 2)
                        ),

                    TextEntry::make('remaining_amount_cents')
                        ->label('Remaining')
                        ->formatStateUsing(
                            fn ($state) => 'NZ$ ' . number_format(((int) ($state ?? 0)) / 100, 2)
                        ),
                ])
                ->columns(4),

            // HOLD STATUS — your existing rich hold info
            InfoSection::make('Hold status')
                ->schema([
                    TextEntry::make('hold_status_text')
                        ->label('Status')
                        ->getStateUsing(fn (Job $record) => $this->holdStateText($record)),

                    TextEntry::make('hold_amount_display')
                        ->label('Hold amount')
                        ->getStateUsing(function (Job $record) {
                            $dep = $this->latestDepositForJob($record);
                            if (! $dep) {
                                return '—';
                            }

                            $cents = (int) ($dep->amount_cents ?? 0);

                            return $cents > 0
                                ? 'NZ$ ' . number_format($cents / 100, 2)
                                : '—';
                        }),

                    TextEntry::make('hold_captured_at')
                        ->label('Captured at')
                        ->getStateUsing(function (Job $record) {
                            $dep = $this->latestDepositForJob($record);
                            $ts  = $dep?->captured_at;

                            if (! $ts) {
                                return '—';
                            }

                            return $ts instanceof \Carbon\CarbonInterface
                                ? $ts->format('Y-m-d H:i')
                                : Carbon::parse($ts)->format('Y-m-d H:i');
                        }),

                    TextEntry::make('hold_planned_cancel_at')
                        ->label('Planned auto-cancel')
                        ->getStateUsing(function (Job $record) {
                            $dep = $this->latestDepositForJob($record);
                            $p   = $dep?->planned_cancel_at;

                            if (! $p) {
                                return '—';
                            }

                            return $p instanceof \Carbon\CarbonInterface
                                ? $p->format('Y-m-d H:i')
                                : Carbon::parse($p)->format('Y-m-d H:i');
                        }),
                ])
                ->columns(4),
        ]);
    }

    // =========================================================================
    // Actions + helpers
    // =========================================================================

    /** Send-now action handler. */
    public function sendPaymentRequest(): void
    {
        /** @var Job $job */
        $job = $this->getRecord();

        try {
            $payload = [];
            [$ok, $msg] = JobController::emailPaymentRequestStatic($job, $payload);

            if ($ok) {
                Notification::make()
                    ->title('Payment request sent')
                    ->body($msg ?: 'An email with the payment link has been sent to the customer.')
                    ->success()
                    ->send();
            } else {
                Notification::make()
                    ->title('Failed to send email')
                    ->body($msg ?: 'Mailer reported a problem. Check logs for details.')
                    ->danger()
                    ->send();
            }
        } catch (\Throwable $e) {
            Log::error('email.payment_request.failed', [
                'job_id' => $job->getKey(),
                'err'    => $e->getMessage(),
            ]);

            Notification::make()
                ->title('Failed to send email')
                ->body($e->getMessage())
                ->danger()
                ->send();
        }
    }

    /** Signed customer pay URL. */
    protected function signedPayUrl(Job $job): string
    {
        return URL::signedRoute('portal.pay.show.job', ['job' => $job->getKey()]);
    }

    /** Public hold page URL (token-based, route-safe). */
    protected function holdPageUrl(Job $job): ?string
    {
        $token = optional($job->booking)->portal_token ?? null;

        if (blank($token)) {
            return null;
        }

        if (! Route::has('portal.pay.hold.page')) {
            return null;
        }

        try {
            return route('portal.pay.hold.page', ['token' => $token]);
        } catch (\Throwable $e) {
            Log::warning('holdPageUrl.failed', [
                'job_id' => $job->getKey(),
                'err'    => $e->getMessage(),
            ]);

            return null;
        }
    }

    /** Latest Deposit for this Job (or null). */
    protected function latestDepositForJob(Job $job): ?Deposit
    {
        return Deposit::query()
            ->where('job_id', $job->getKey())
            ->latest('id')
            ->first();
    }

    // =========================================================================
    // Status + Reminder helpers
    // =========================================================================

    /** Tooltip to explain why hold actions are disabled. */
    protected function holdDisabledTooltip(): string
    {
        /** @var Job $job */
        $job     = $this->getRecord();
        $planned = (int) (optional($job->flow)->hold_amount_cents ?? 0);

        if ($planned <= 0) {
            return 'No hold required by this flow.';
        }

        return 'Hold page not available yet (missing portal token or route).';
    }

    /** Non-fatal notification if start is within N days and hold needs doing. */
    protected function maybeNotifyHoldReminder(): void
    {
        /** @var Job $job */
        $job = $this->getRecord();

        $start = $job?->start_at;
        if (! $start instanceof \Carbon\CarbonInterface) {
            return;
        }

        if (! $this->needsHoldAction($job)) {
            return;
        }

        $now = now();

        if ($start->greaterThan($now) && $start->diffInDays($now) <= $this->holdReminderDays) {
            $stateText = $this->holdStateText($job);

            Notification::make()
                ->title('Hold authorisation required')
                ->body(
                    'Reservation is approaching (' .
                    $start->timezone(config('app.timezone'))->toDayDateTimeString() .
                    "). Status: {$stateText}. Use the hold buttons to get this done."
                )
                ->warning()
                ->send();
        }
    }

    /** Whether hold work is still required. */
    protected function needsHoldAction(Job $job): bool
    {
        $state = $this->holdState($job);

        return in_array($state, [
            'not_created',
            'needs_authorization',
            'authorized_not_captured',
            'pending',
        ], true);
    }

    /** Machine hold state for UI. */
    protected function holdState(Job $job): string
    {
        $dep = $this->latestDepositForJob($job);

        // No deposit yet. If Flow expects a hold, call it "not_created".
        if (! $dep) {
            $planned = (int) (optional($job->flow)->hold_amount_cents ?? 0);

            return $planned > 0 ? 'not_created' : 'none';
        }

        if (! empty($dep->captured_at)) {
            return 'captured';
        }

        $status = (string) $dep->status;

        // Customer action needed (e.g., 3DS / SCA)
        if (in_array($status, ['requires_action', 'requires_confirmation'], true)) {
            return 'needs_authorization';
        }

        // Already authorised, waiting for capture
        if (in_array($status, ['requires_capture', 'authorized'], true)) {
            return 'authorized_not_captured';
        }

        // Still pending / processing upstream
        if (in_array($status, ['pending', 'processing'], true)) {
            return 'pending';
        }

        return 'none';
    }

    /** Human text for status. */
    protected function holdStateText(Job $job): string
    {
        return match ($this->holdState($job)) {
            'not_created'             => 'Not created',
            'needs_authorization'     => 'Needs customer authorisation',
            'authorized_not_captured' => 'Authorised — not captured',
            'pending'                 => 'Pending',
            'captured'                => 'Captured',
            default                   => 'No hold',
        };
    }
}
