<?php

namespace App\Filament\Resources\JobResource\Widgets;

use App\Models\Deposit;
use App\Models\Job;
use Filament\Widgets\Widget;
use Illuminate\Support\Carbon;

class HoldReleaseNotice extends Widget
{
    protected static string $view = 'filament.resources.job.hold-release-notice';
    protected static bool $isLazy = false;
    protected int|string|array $columnSpan = 'full';

    public ?Job $record = null;

    protected function getViewData(): array
    {
        return ['message' => $this->computeMessage($this->record)];
    }

    protected function computeMessage(?Job $job): ?string
    {
        if (!$job) return null;

        // Latest authorized deposit for this job
        $deposit = Deposit::query()
            ->where('job_id', $job->getKey())
            ->whereIn('status', ['authorized', 'authorised'])
            ->orderByDesc('created_at')
            ->orderByDesc('id')
            ->first();

        if (!$deposit) return null;

        // --- config: prefer Job; fall back to Flow ---
        $flow = method_exists($job, 'flow') ? $job->flow : null;

        $days = (int) (
            $job->auto_release_after_return_days
            ?? $job->auto_release_after_days
            ?? $job->auto_release_days
            ?? ($flow->auto_release_after_return_days ?? $flow->auto_release_after_days ?? $flow->auto_release_days ?? 0)
        );

        $enabled = (bool) (
            $job->auto_cancel_if_no_capture
            ?? $job->auto_cancel
            ?? ($flow->auto_cancel_if_no_capture ?? $flow->auto_cancel ?? true)
        );

        $endRaw = $job->end_at ?? $job->endAt ?? null;
        if (!$enabled || $days <= 0 || !$endRaw) return null;

        $tz       = $job->timezone ?? $flow->timezone ?? config('app.timezone', 'UTC');
        $endAt    = Carbon::parse($endRaw)->timezone($tz);
        $release  = $endAt->copy()->addDays($days);

        $fmtRelease = $release->isoFormat('ddd D MMM, HH:mm z');
        $fmtEnd     = $endAt->isoFormat('ddd D MMM, HH:mm z');

        return "Hold will be automatically released on {$fmtRelease} (return: {$fmtEnd} + {$days} day" . ($days === 1 ? '' : 's') . ')';
    }
}
