<?php

declare(strict_types=1);

namespace App\Filament\Resources;

use App\Filament\Resources\OwnerResource\Pages;
use App\Filament\Resources\OwnerResource\RelationManagers\LedgerEntriesRelationManager;
use App\Filament\Resources\OwnerResource\RelationManagers\PayoutsRelationManager;
use App\Filament\Resources\OwnerResource\RelationManagers\VehiclesRelationManager;
use App\Models\Owner;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class OwnerResource extends Resource
{
    protected static ?string $model = Owner::class;

    protected static ?string $navigationIcon  = 'heroicon-o-user-group';
    protected static ?string $navigationGroup = 'Fleet';
    protected static ?string $navigationLabel = 'Consignment Owners';
    protected static ?int    $navigationSort  = 5;

    public static function getModelLabel(): string
    {
        return 'Owner';
    }

    public static function getPluralModelLabel(): string
    {
        return 'Owners';
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make('Owner details')
                    ->columns(2)
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('Name')
                            ->required()
                            ->maxLength(255),

                        Forms\Components\TextInput::make('email')
                            ->label('Email')
                            ->email()
                            ->maxLength(255),

                        Forms\Components\TextInput::make('phone')
                            ->label('Phone')
                            ->maxLength(255),

                        Forms\Components\Textarea::make('notes')
                            ->label('Notes')
                            ->columnSpanFull(),
                    ]),

                Forms\Components\Section::make('Payment / Bank details')
    ->columns(2)
    ->schema([
        Forms\Components\TextInput::make('legal_name')
            ->label('Legal name')
            ->maxLength(255)
            ->helperText('Used on statements; defaults to Name if left blank.'),

        Forms\Components\TextInput::make('bank_account_name')
            ->label('Bank account name')
            ->maxLength(255),

        Forms\Components\TextInput::make('bank_account_number')
            ->label('Bank account number')
            ->maxLength(255),

        Forms\Components\TextInput::make('bank_reference')
            ->label('Bank reference')
            ->maxLength(255)
            ->helperText('Reference that will appear on their bank statement.'),

        Forms\Components\Select::make('payout_schedule')
            ->label('Payout schedule')
            ->options([
                'weekly'      => 'Weekly',
                'fortnightly' => 'Fortnightly',
                'monthly'     => 'Monthly',
                'ad-hoc'      => 'Ad-hoc',
            ])
            ->default('ad-hoc'),

        Forms\Components\TextInput::make('cc_email')
            ->label('CC email for statements')
            ->email()
            ->maxLength(255),

        Forms\Components\Toggle::make('gst_registered')
            ->label('GST registered?'),

        Forms\Components\TextInput::make('gst_number')
            ->label('GST number')
            ->maxLength(255)
            ->visible(fn (callable $get): bool => (bool) $get('gst_registered')),

        // Keep your existing free-form field as optional notes if you still want it
        Forms\Components\Textarea::make('bank_account_details')
            ->label('Bank account details (notes)')
            ->rows(3)
            ->columnSpanFull()
            ->hint('Optional free-form notes e.g. legacy payout instructions.'),
    ]),

            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Owner')
                    ->searchable()
                    ->sortable(),

                Tables\Columns\TextColumn::make('email')
                    ->label('Email')
                    ->toggleable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('phone')
                    ->label('Phone')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('vehicles_count')
                    ->label('Vehicles')
                    ->counts('vehicles')
                    ->sortable(),

                Tables\Columns\TextColumn::make('earnings_cents')
                    ->label('Earnings')
                    ->state(fn (Owner $record): int => $record->earnings_cents)
                    ->formatStateUsing(fn (int $state): string => sprintf('$%0.2f', $state / 100))
                    ->sortable()
                    ->alignment('right')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('costs_cents')
                    ->label('Costs')
                    ->state(fn (Owner $record): int => $record->costs_cents)
                    ->formatStateUsing(fn (int $state): string => sprintf('$%0.2f', $state / 100))
                    ->sortable()
                    ->alignment('right')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('payouts_cents')
                    ->label('Payouts')
                    ->state(fn (Owner $record): int => $record->payouts_cents)
                    ->formatStateUsing(fn (int $state): string => sprintf('$%0.2f', $state / 100))
                    ->sortable()
                    ->alignment('right')
                    ->toggleable(),

                Tables\Columns\TextColumn::make('balance')
                    ->label('Balance')
                    ->state(fn (Owner $record): int => $record->balance_cents)
                    ->formatStateUsing(fn (int $state): string => sprintf('$%0.2f', $state / 100))
                    ->sortable()
                    ->alignment('right')
                    ->description('Earnings + adjustments - costs - payouts'),
            ])
            ->defaultSort('name')
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    /** @return array<class-string> */
    public static function getRelations(): array
    {
        return [
            VehiclesRelationManager::class,
            LedgerEntriesRelationManager::class,
            PayoutsRelationManager::class,
        ];
    }

    /** @return array<string, class-string> */
public static function getPages(): array
{
    return [
        'index'     => Pages\ListOwners::route('/'),
        'create'    => Pages\CreateOwner::route('/create'),
        'view'      => Pages\ViewOwner::route('/{record}'),
        'edit'      => Pages\EditOwner::route('/{record}/edit'),
        'statement' => Pages\OwnerStatement::route('/{record}/statement'),
    ];
}


    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()
            ->withCount('vehicles')
            // Preload sums by type so the accessors can use them without N+1 queries
            ->withSum([
                'ledgerEntries as earnings_cents' => function (Builder $query): void {
                    $query->where('type', 'earning');
                },
            ], 'amount_cents')
            ->withSum([
                'ledgerEntries as costs_cents' => function (Builder $query): void {
                    $query->where('type', 'cost');
                },
            ], 'amount_cents')
            ->withSum([
                'ledgerEntries as payouts_cents' => function (Builder $query): void {
                    $query->where('type', 'payout');
                },
            ], 'amount_cents')
            ->withSum([
                'ledgerEntries as adjustments_cents' => function (Builder $query): void {
                    $query->where('type', 'adjustment');
                },
            ], 'amount_cents');
    }
}
