<?php

declare(strict_types=1);

namespace App\Filament\Resources\OwnerResource\Pages;

use App\Filament\Resources\OwnerResource;
use App\Mail\OwnerStatementMail;
use App\Models\Job;
use App\Models\Owner;
use App\Models\OwnerLedgerEntry;
use Barryvdh\DomPDF\Facade\Pdf;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\Page;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Mail;

class OwnerStatement extends Page
{
    protected static string $resource = OwnerResource::class;

    protected static string $view = 'filament.resources.owner-resource.pages.owner-statement';

    public Owner $owner;

    public ?string $startDate = null;
    public ?string $endDate = null;

    public function mount(int|string $record): void
    {
        $this->owner = Owner::findOrFail($record);

        $this->startDate ??= now()->startOfMonth()->toDateString();
        $this->endDate ??= now()->endOfDay()->toDateString();
    }

    public function getEntries(): Collection
    {
        $query = OwnerLedgerEntry::query()
            ->where('owner_id', $this->owner->id)
            ->with(['vehicle', 'job'])
            ->orderBy('occurred_at');

        if ($this->startDate) {
            $query->whereDate('occurred_at', '>=', $this->startDate);
        }

        if ($this->endDate) {
            $query->whereDate('occurred_at', '<=', $this->endDate);
        }

        return $query->get();
    }

    public function getTotals(): array
    {
        $entries = $this->getEntries();

        $totals = [
            'earnings_cents'    => 0,
            'costs_cents'       => 0,
            'payouts_cents'     => 0,
            'adjustments_cents' => 0,
            'balance_cents'     => 0,
        ];

        foreach ($entries as $entry) {
            $typeEnum = $entry->type;
            $type = $typeEnum instanceof \BackedEnum ? $typeEnum->value : (string) $typeEnum;

            $rawCents     = (int) $entry->amount_cents;
            $ownerPercent = $entry->vehicle?->owner_share_percent;

            $ownerCents = $rawCents;

            if ($type === 'earning' && $ownerPercent !== null) {
                $ownerCents = (int) round($rawCents * ($ownerPercent / 100));
            } elseif (in_array($type, ['cost', 'payout'], true)) {
                $ownerCents = $rawCents;
            }

            switch ($type) {
                case 'earning':
                    $totals['earnings_cents'] += $ownerCents;
                    break;
                case 'cost':
                    $totals['costs_cents'] += $ownerCents;
                    break;
                case 'payout':
                    $totals['payouts_cents'] += $ownerCents;
                    break;
                default:
                    $totals['adjustments_cents'] += $ownerCents;
                    break;
            }
        }

        $totals['balance_cents'] =
            $totals['earnings_cents']
            + $totals['adjustments_cents']
            - $totals['costs_cents']
            - $totals['payouts_cents'];

        return $totals;
    }

    public function emailStatement(): void
    {
        $owner   = $this->owner;
        $entries = $this->getEntries();
        $totals  = $this->getTotals();

        $from = $this->startDate
            ? Carbon::parse($this->startDate)
            : ($entries->min('occurred_at')
                ? Carbon::parse($entries->min('occurred_at'))->startOfDay()
                : now()->startOfMonth());

        $to = $this->endDate
            ? Carbon::parse($this->endDate)
            : ($entries->max('occurred_at')
                ? Carbon::parse($entries->max('occurred_at'))->endOfDay()
                : now());

        $statementId = $owner->id . '-' . now()->format('YmdHis');

        // Vehicle names included in this period
        $vehicles = $entries
            ->pluck('vehicle.name')
            ->filter()
            ->unique()
            ->values();

        // Upcoming payout for THIS statement (already in your layout)
        $payoutAmountCents = (int) ($totals['balance_cents'] ?? 0);

        // ---------------------------------------------------------------------
        // Future earnings preview (next month's bookings)
        // Rule: pay on the 10th of each month for bookings that ended in the
        // previous calendar month.
        // So: statement period = November -> future preview = December bookings,
        // to be paid on 10 January.
        // ---------------------------------------------------------------------

        // The month immediately after this statement's "to" date.
        $previewMonthStart = $to->copy()->addMonthNoOverflow()->startOfMonth();
        $previewMonthEnd   = $previewMonthStart->copy()->endOfMonth();

        // Payout date for those bookings: 10th of the month AFTER preview month.
        $previewPayoutDate = $previewMonthStart
            ->copy()
            ->addMonthNoOverflow()
            ->day(10)
            ->startOfDay();

        $vehicleIds = $owner->vehicles()->pluck('id');

        $previewJobs = Job::query()
            ->whereIn('vehicle_id', $vehicleIds)
            ->whereBetween('end_at', [$previewMonthStart, $previewMonthEnd])
            ->with(['vehicle'])
            ->orderBy('end_at')
            ->get();

        $previewTotals = [
            'month_label'        => $previewMonthStart->format('F Y'),
            'owner_total_cents'  => 0,
            'per_vehicle_cents'  => [],
        ];

        foreach ($previewJobs as $job) {
            $vehicleName = $job->vehicle->name ?? 'Unassigned';

            // Best-guess gross amount field; adjust if your Job model differs.
            $grossCents = (int) ($job->charge_amount_cents ?? $job->total_price_cents ?? 0);
            $ownerPercent = $job->vehicle?->owner_share_percent;

            $ownerCents = $ownerPercent !== null
                ? (int) round($grossCents * ($ownerPercent / 100))
                : $grossCents;

            $previewTotals['owner_total_cents'] += $ownerCents;

            if (! isset($previewTotals['per_vehicle_cents'][$vehicleName])) {
                $previewTotals['per_vehicle_cents'][$vehicleName] = 0;
            }
            $previewTotals['per_vehicle_cents'][$vehicleName] += $ownerCents;
        }

        // ---------------------------------------------------------------------

        $pdf = Pdf::loadView('pdf.owner_statement', [
            'owner'              => $owner,
            'from'               => $from,
            'to'                 => $to,
            'entries'            => $entries,
            'totals'             => $totals,
            'statementId'        => $statementId,
            'payoutAmountCents'  => $payoutAmountCents,
            'vehicles'           => $vehicles,
            'previewTotals'      => $previewTotals,
            'previewPayoutDate'  => $previewPayoutDate,
        ]);

        $pdfData  = $pdf->output();
        $fileName = 'owner-statement-' . $statementId . '.pdf';

        $toAddress = $owner->email;

        if (! $toAddress) {
            Notification::make()
                ->title('No email on file')
                ->body('This owner does not have an email address set.')
                ->danger()
                ->send();

            return;
        }

        $cc = [];
        if (! empty($owner->cc_email)) {
            $cc[] = $owner->cc_email;
        }

        Mail::to($toAddress)
            ->cc($cc)
            ->send(new OwnerStatementMail(
                owner: $owner,
                fromDate: $from,
                toDate: $to,
                totals: $totals,
                payoutAmountCents: $payoutAmountCents,
                vehicles: $vehicles,
                statementId: $statementId,
                previewTotals: $previewTotals,
                previewPayoutDate: $previewPayoutDate,
                pdfData: $pdfData,
                pdfFileName: $fileName,
            ));

        Notification::make()
            ->title('Statement emailed')
            ->body("Statement emailed to {$toAddress}" . (count($cc) ? ' (CC: ' . implode(', ', $cc) . ')' : ''))
            ->success()
            ->send();
    }
}
