<?php

declare(strict_types=1);

namespace App\Filament\Resources\OwnerResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;

class PayoutsRelationManager extends RelationManager
{
    protected static string $relationship = 'payouts';

    protected static ?string $title = 'Payouts';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('amount_cents')
                    ->label('Amount (cents)')
                    ->numeric()
                    ->required(),

                Forms\Components\TextInput::make('currency_code')
                    ->default('NZD')
                    ->maxLength(3)
                    ->required(),

                Forms\Components\TextInput::make('payout_method')
                    ->default('bank_transfer')
                    ->maxLength(255),

                Forms\Components\TextInput::make('reference')
                    ->maxLength(255)
                    ->helperText('E.g. bank reference, month, etc.'),

                Forms\Components\DateTimePicker::make('paid_at')
                    ->label('Paid at')
                    ->nullable(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('paid_at')
                    ->label('Paid at')
                    ->dateTime()
                    ->sortable(),

                Tables\Columns\TextColumn::make('amount_cents')
                    ->label('Amount')
                    ->alignment('right')
                    ->formatStateUsing(fn (int $state): string => sprintf('$%0.2f', $state / 100)),

                Tables\Columns\TextColumn::make('currency_code')
                    ->label('Cur'),

                Tables\Columns\TextColumn::make('payout_method')
                    ->label('Method'),

                Tables\Columns\TextColumn::make('reference')
                    ->label('Reference')
                    ->limit(40)
                    ->wrap(),
            ])
            ->defaultSort('paid_at', 'desc')
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Record payout'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }
}
