<?php

declare(strict_types=1);

namespace App\Filament\Resources\OwnerResource\RelationManagers;

use App\Models\Vehicle;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;

class VehiclesRelationManager extends RelationManager
{
    protected static string $relationship = 'vehicles';

    protected static ?string $title = 'Vehicles';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                // Typically vehicles are managed via VehicleResource, so keep this minimal.
                Forms\Components\TextInput::make('name')
                    ->label('Name')
                    ->maxLength(255),

                Forms\Components\TextInput::make('registration')
                    ->label('Rego')
                    ->maxLength(255),

                Forms\Components\Select::make('ownership_type')
                    ->label('Ownership type')
                    ->options([
                        'company_owned' => 'Company owned',
                        'consignment'   => 'Consignment',
                    ])
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('name')
            ->columns([
                Tables\Columns\TextColumn::make('name')
                    ->label('Vehicle')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('registration')
                    ->label('Rego')
                    ->sortable()
                    ->searchable(),

                Tables\Columns\TextColumn::make('ownership_type')
                    ->label('Ownership')
                    ->badge()
                    ->formatStateUsing(function (?string $state): string {
                        return match ($state) {
                            'company_owned' => 'Company owned',
                            'consignment'   => 'Consignment',
                            default         => $state ?? '-',
                        };
                    }),

                Tables\Columns\TextColumn::make('maintenance_cost_charged')
                    ->label('Maint. charged')
                    ->state(function (Vehicle $record): int {
                        // sum() returns a string from the DB, so cast to int for strict_types
                        return (int) $record->ownerLedgerEntries()
                            ->where('type', 'cost')
                            ->sum('amount_cents');
                    })
                    ->formatStateUsing(function (?int $state): string {
                        if (! $state) {
                            return '—';
                        }

                        return '$' . number_format($state / 100, 2);
                    })
                    ->sortable(false),

                Tables\Columns\TextColumn::make('jobs_count')
                    ->counts('jobs')
                    ->label('Bookings')
                    ->sortable(),
            ])
            ->defaultSort('name')
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->label('Add vehicle'),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\ViewAction::make()
                    ->url(fn (Vehicle $record): string => route('filament.admin.resources.vehicles.edit', $record)),
            ])
            ->bulkActions([]);
    }
}
