<?php

declare(strict_types=1);

namespace App\Filament\Resources;

use App\Filament\Resources\UserResource\Pages;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Validation\Rules\Password;

class UserResource extends Resource
{
    protected static ?string $model = User::class;

    protected static ?string $navigationIcon  = 'heroicon-o-users';
    protected static ?string $navigationGroup = 'Settings';
    protected static ?string $navigationLabel = 'Users';
    protected static ?int    $navigationSort  = 99;

    public static function form(Form $form): Form
    {
        return $form->schema([
            Section::make('User details')
                ->schema([
                    TextInput::make('name')
                        ->required()
                        ->maxLength(120),

                    TextInput::make('email')
                        ->email()
                        ->required()
                        ->unique(ignoreRecord: true),

                    Select::make('role')
                        ->label('Role')
                        ->options([
                            User::ROLE_ADMIN => 'Admin',
                            User::ROLE_STAFF => 'Staff',
                        ])
                        ->default(User::ROLE_STAFF)
                        ->required(),

                    TextInput::make('password')
                        ->password()
                        ->revealable()
                        ->rule(Password::default())
                        ->required(fn (string $operation): bool => $operation === 'create')
                        ->dehydrateStateUsing(
                            fn (?string $state): ?string => filled($state) ? bcrypt($state) : null
                        )
                        ->dehydrated(fn (?string $state): bool => filled($state)),

                    TextInput::make('password_confirmation')
                        ->password()
                        ->revealable()
                        ->label('Confirm password')
                        ->same('password')
                        ->required(fn (string $operation): bool => $operation === 'create'),
                ])
                ->columns(2),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('email')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('role')
                    ->label('Role')
                    ->badge()
                    ->sortable()
                    ->colors([
                        'success'   => fn (string $state): bool => $state === User::ROLE_ADMIN,
                        'secondary' => fn (string $state): bool => $state === User::ROLE_STAFF,
                    ])
                    ->formatStateUsing(fn (string $state): string => ucfirst($state)),

                TextColumn::make('created_at')
                    ->dateTime()
                    ->since()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index'  => Pages\ListUsers::route('/'),
            'create' => Pages\CreateUser::route('/create'),
            'edit'   => Pages\EditUser::route('/{record}/edit'),
        ];
    }
}
