<?php

declare(strict_types=1);

namespace App\Filament\Widgets;

use App\Models\Deposit;
use App\Models\Job;
use App\Models\Payment;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Card;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class PaymentStatsWidget extends BaseWidget
{
    // Non-static for StatsOverviewWidget in Filament v3
    protected ?string $heading = 'Key Stats';

    protected function getCards(): array
    {
        $today = Carbon::today();
        $now   = Carbon::now();
        $last7 = $now->clone()->subDays(7);
        $startOfWeek = $now->clone()->startOfWeek();

        // ---------------- Payments ----------------
        $paymentsBase = Payment::query();
        if (Schema::hasColumn('payments', 'status')) {
            $paymentsBase->whereIn('status', ['succeeded', 'paid', 'captured']);
        }

        $paymentsToday = (clone $paymentsBase)->whereDate('created_at', $today);
        $payments7Days = (clone $paymentsBase)->where('created_at', '>=', $last7);

        $sumCents = function ($q): int {
            if (Schema::hasColumn('payments', 'amount_cents')) {
                return (int) $q->sum('amount_cents');
            }
            if (Schema::hasColumn('payments', 'amount')) {
                return (int) round((float) $q->sum(DB::raw('COALESCE(amount,0)')) * 100);
            }
            return 0;
        };

        $countPaymentsToday = (int) (clone $paymentsToday)->count();
        $sumPaymentsTodayC  = $sumCents($paymentsToday);

        $countPayments7     = (int) (clone $payments7Days)->count();
        $sumPayments7C      = $sumCents($payments7Days);

        // ---------------- Deposits / Holds ----------------
        $holdsAuthorizedToday = Deposit::query()
            ->when(Schema::hasColumn('deposits', 'status'), fn ($q) => $q->where('status', 'authorized'))
            ->whereDate(Schema::hasColumn('deposits', 'authorized_at') ? 'authorized_at' : 'created_at', $today)
            ->count();

        $capturedThisWeekC = (int) Deposit::query()
            ->when(Schema::hasColumn('deposits', 'status'), fn ($q) => $q->where('status', 'captured'))
            ->when(
                Schema::hasColumn('deposits', 'captured_at'),
                fn ($q) => $q->where('captured_at', '>=', $startOfWeek),
                fn ($q) => $q->where('updated_at', '>=', $startOfWeek)
            )
            ->when(
                Schema::hasColumn('deposits', 'amount_cents'),
                fn ($q) => $q->selectRaw('COALESCE(SUM(amount_cents),0) as s'),
                fn ($q) => $q->selectRaw('ROUND(COALESCE(SUM(amount),0)*100) as s')
            )
            ->value('s') ?? 0;

        $holdsReleasedToday = Deposit::query()
            ->when(Schema::hasColumn('deposits', 'status'), fn ($q) => $q->where('status', 'released'))
            ->whereDate(Schema::hasColumn('deposits', 'released_at') ? 'released_at' : 'updated_at', $today)
            ->count();

        // ---------------- Jobs ----------------
        $activeJobs = Job::query()
            ->when(Schema::hasColumn('jobs', 'status'), fn ($q) => $q->where('status', 'active'))
            ->count();

        $jobsPaidToday = Job::query()
            ->when(
                Schema::hasColumn('jobs', 'paid_at'),
                fn ($q) => $q->whereDate('paid_at', $today),
                fn ($q) => $q->when(Schema::hasColumn('jobs', 'status'),
                    fn ($qq) => $qq->where('status', 'paid')->whereDate('updated_at', $today),
                    fn ($qq) => $qq->whereDate('updated_at', $today)
                )
            )
            ->count();

        // Unpaid Jobs (awaiting payment) – count + value
        $unpaidQuery = Job::query();
        if (Schema::hasColumn('jobs', 'paid_at')) {
            $unpaidQuery->whereNull('paid_at');
        } elseif (Schema::hasColumn('jobs', 'status')) {
            $unpaidQuery->where('status', '!=', 'paid');
        }

        // Only include jobs with amount due
        $sumUnpaidCents = 0;
        if (Schema::hasColumn('jobs', 'charge_amount_cents')) {
            $unpaidQuery->where('charge_amount_cents', '>', 0);
            $sumUnpaidCents = (int) $unpaidQuery->sum('charge_amount_cents');
        } elseif (Schema::hasColumn('jobs', 'amount_due_cents')) {
            $unpaidQuery->where('amount_due_cents', '>', 0);
            $sumUnpaidCents = (int) $unpaidQuery->sum('amount_due_cents');
        } elseif (Schema::hasColumn('jobs', 'charge_amount')) {
            $unpaidQuery->where('charge_amount', '>', 0);
            $sumUnpaidCents = (int) round((float) $unpaidQuery->sum('charge_amount') * 100);
        }
        $unpaidJobs = (int) $unpaidQuery->count();

        // ---------------- Helpers ----------------
        $money = fn (int $cents): string => '$' . number_format($cents / 100, 2);

        return [
            Card::make('Payments Today', $money($sumPaymentsTodayC))
                ->description($countPaymentsToday . ' completed')
                ->icon('heroicon-m-currency-dollar'),

            Card::make('Payments (Last 7 Days)', $money($sumPayments7C))
                ->description($countPayments7 . ' completed')
                ->icon('heroicon-m-chart-bar'),

            Card::make('Captured Deposits (This Week)', $money($capturedThisWeekC))
                ->description('Holds converted to charges')
                ->icon('heroicon-m-lock-closed'),

            Card::make('New Holds Authorized Today', (string) $holdsAuthorizedToday)
                ->description('Awaiting capture or release')
                ->icon('heroicon-m-check-badge'),

            Card::make('Holds Released Today', (string) $holdsReleasedToday)
                ->description('Returned to customers')
                ->icon('heroicon-m-arrow-uturn-left'),

            Card::make('Active Jobs', (string) $activeJobs)
                ->description('Currently in progress')
                ->icon('heroicon-m-briefcase'),

            Card::make('Jobs Marked Paid Today', (string) $jobsPaidToday)
                ->description('Status changed to paid')
                ->icon('heroicon-m-check-circle'),

            // New card with count + value
            Card::make('Unpaid Jobs', (string) $unpaidJobs)
                ->description('Value: ' . $money($sumUnpaidCents))
                ->icon('heroicon-m-clock'),
        ];
    }
}
