<?php

declare(strict_types=1);

namespace App\Filament\Widgets;

use App\Models\Payment;
use Filament\Widgets\ChartWidget;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class PaymentsTrendChart extends ChartWidget
{
    // Chart title
    protected static ?string $heading = 'Payments – Last 7 Days';

    // ✅ Make this widget span the full dashboard width
    protected int|array|string $columnSpan = 'full';

    // (Optional) control height if needed
    // protected static ?string $maxHeight = '320px';

    protected function getData(): array
    {
        $labels = [];
        $dataDollars = [];

        for ($i = 6; $i >= 0; $i--) {
            $day = Carbon::today()->subDays($i);
            $labels[] = $day->format('D');

            $q = Payment::query()->whereDate('created_at', $day);

            // Only count successful/settled payments if the column exists
            if (Schema::hasColumn('payments', 'status')) {
                $q->whereIn('status', ['succeeded', 'paid', 'captured']);
            }

            // Prefer cents if present; otherwise fall back to dollar column
            if (Schema::hasColumn('payments', 'amount_cents')) {
                $sumCents = (int) $q->sum('amount_cents');
            } elseif (Schema::hasColumn('payments', 'amount')) {
                $sumCents = (int) round((float) $q->sum(DB::raw('COALESCE(amount,0)')) * 100);
            } else {
                $sumCents = 0;
            }

            $dataDollars[] = round($sumCents / 100, 2);
        }

        return [
            'datasets' => [
                [
                    'label' => 'Payments ($)',
                    'data' => $dataDollars,
                    // You can add styling options here if you want
                ],
            ],
            'labels' => $labels,
        ];
    }

    protected function getType(): string
    {
        return 'line';
    }
}
