<?php

namespace App\Filament\Widgets;

use App\Models\Payment;
use Carbon\CarbonImmutable;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Widgets\TableWidget as BaseWidget;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Schema;

class UpcomingHoldReleases extends BaseWidget
{
    protected static ?string $heading = 'Upcoming Bond Hold Releases';

    /** Full-width and pinned to the top of the dashboard */
    protected int|string|array $columnSpan = 'full';
    protected static ?int $sort = -100;

    /** How many days ahead to show (available for future filtering if needed) */
    protected int $lookaheadDays = 14;

    /** Base query for open bond holds with "hold-ish" status */
    protected function baseHoldQuery(): Builder
    {
        $q = Payment::query()
            ->with('job')
            ->where('type', 'bond_hold')
            ->whereNotIn('status', ['succeeded', 'captured', 'canceled', 'failed'])
            ->where(function (Builder $i) {
                $i->whereIn('status', ['pending', 'authorized', 'requires_capture']);

                if (Schema::hasColumn('payments', 'stripe_payment_intent_status')) {
                    $i->orWhere('stripe_payment_intent_status', 'requires_capture');
                }
                if (Schema::hasColumn('payments', 'capture_method')) {
                    $i->orWhere('capture_method', 'manual');
                }
                if (Schema::hasColumn('payments', 'is_hold')) {
                    $i->orWhere('is_hold', true);
                }
            });

        // Light ordering (urgency is conveyed via the countdown color/bold)
        return $q->latest('created_at');
    }

    /**
     * Compute a "due_at" with sensible fallbacks:
     *   1) payments.auto_release_at
     *   2) payments.hold_expires_at (Stripe auth expiry)
     *   3) job.return_at / job.end_at / job.dropoff_at
     *   4) created_at + 7 days (Stripe default auth window)
     */
    protected function computeDueAt(Payment $p): \Carbon\CarbonInterface
    {
        if (Schema::hasColumn('payments', 'auto_release_at') && $p->auto_release_at) {
            return CarbonImmutable::parse($p->auto_release_at);
        }
        if (Schema::hasColumn('payments', 'hold_expires_at') && $p->hold_expires_at) {
            return CarbonImmutable::parse($p->hold_expires_at);
        }

        $job = $p->relationLoaded('job') ? $p->job : $p->job()->first();
        if ($job) {
            foreach (['return_at', 'end_at', 'dropoff_at'] as $field) {
                if (! empty($job->{$field})) {
                    return CarbonImmutable::parse($job->{$field});
                }
            }
        }

        // Fallback: 7 days after creation
        return CarbonImmutable::parse($p->created_at)->addDays(7);
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(fn () => $this->baseHoldQuery())
            ->paginated(false)
            ->emptyStateHeading('No upcoming bond hold releases')
            ->columns([
                Tables\Columns\TextColumn::make('job.external_reference')
                    ->label('Job')
                    ->formatStateUsing(fn ($state) => $state ?? '—')
                    ->searchable(),

                Tables\Columns\TextColumn::make('amount_cents')
                    ->label('Amount')
                    ->formatStateUsing(fn ($state, Payment $r) =>
                        ($r->currency ?? 'NZD') . ' ' . number_format(((int) $state) / 100, 2)
                    )
                    ->alignRight(),

                Tables\Columns\TextColumn::make('status')
                    ->label('Status')
                    ->badge()
                    ->colors([
                        'warning' => ['pending', 'requires_capture'],
                        'info'    => ['authorized'],
                    ]),

                // DAYS ONLY; bold + red if overdue or within 2 days
                Tables\Columns\TextColumn::make('countdown')
                    ->label('Due in')
                    ->getStateUsing(function (Payment $r) {
                        $due = $this->computeDueAt($r);
                        $now = CarbonImmutable::now();

                        if ($due->isPast()) {
                            return 'Overdue';
                        }

                        $hours = $now->diffInHours($due);
                        $days  = (int) ceil($hours / 24); // 1.2d -> 2d, 0.1d -> 1d
                        return "{$days}d";
                    })
                    ->badge()
                    ->color(function (Payment $r) {
                        $due = $this->computeDueAt($r);
                        if ($due->isPast()) {
                            return 'danger'; // red
                        }
                        return CarbonImmutable::now()->diffInHours($due) <= 48 ? 'danger' : 'gray';
                    })
                    ->extraAttributes(function (Payment $r) {
                        $due = $this->computeDueAt($r);
                        $urgent = $due->isPast() || CarbonImmutable::now()->diffInHours($due) <= 48;
                        return $urgent ? ['class' => 'font-bold'] : [];
                    }),

                // Exact timestamp (hidden by default)
                Tables\Columns\TextColumn::make('due_at')
                    ->label('Release at')
                    ->getStateUsing(fn (Payment $r) => $this->computeDueAt($r)->toDayDateTimeString())
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->actions([
                Action::make('capture')
                    ->label('Capture')
                    ->icon('heroicon-o-lock-closed')
                    ->color('success')
                    ->requiresConfirmation()
                    ->action(function (Payment $p) {
                        if (method_exists($p, 'captureHold')) {
                            $p->captureHold();
                        } elseif (method_exists($p, 'capture')) {
                            $p->capture();
                        }
                        $p->refresh();
                    }),

                Action::make('release')
                    ->label('Release')
                    ->icon('heroicon-o-lock-open')
                    ->color('danger')
                    ->requiresConfirmation()
                    ->action(function (Payment $p) {
                        if (method_exists($p, 'releaseHold')) {
                            $p->releaseHold();
                        } elseif (method_exists($p, 'cancel')) {
                            $p->cancel();
                        }
                        $p->refresh();
                    }),

                Action::make('open')
                    ->label('Open')
                    ->icon('heroicon-o-arrow-top-right-on-square')
                    ->url(fn (Payment $p) => route('filament.admin.resources.payments.view', ['record' => $p]))
                    ->openUrlInNewTab(),
            ]);
    }
}
