<?php

declare(strict_types=1);

namespace App\Filament\Widgets;

use App\Filament\Resources\VehicleResource;
use App\Filament\Resources\VehicleMaintenanceLogResource;
use App\Models\Vehicle;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Widgets\TableWidget as BaseWidget;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class UpcomingMaintenanceTodoList extends BaseWidget
{
    protected static ?string $heading = 'Vehicle Maintenance To-Do';

    protected int|string|array $columnSpan = 'full';

    public function table(Table $table): Table
    {
        return $table
            ->query($this->getTableQuery())
            ->columns([
                // Vehicle name/label
                Tables\Columns\TextColumn::make('name')
                    ->label('Vehicle')
                    ->searchable(),

                // Next booking / availability
                Tables\Columns\TextColumn::make('next_booking')
                    ->label('Next booking')
                    ->getStateUsing(fn (Vehicle $record): string => $this->resolveNextBooking($record)['label'])
                    ->description(fn (Vehicle $record): ?string => $this->resolveNextBooking($record)['description'])
                    ->badge()
                    ->color(function (Vehicle $record): string {
                        return match ($this->resolveNextBooking($record)['status']) {
                            'current' => 'danger',  // on hire now
                            'soon'    => 'warning', // upcoming soon
                            'free'    => 'success', // good window / no bookings
                            default   => 'gray',
                        };
                    }),

                // Overall urgency label
                Tables\Columns\TextColumn::make('urgency')
                    ->label('Urgency')
                    ->badge()
                    ->getStateUsing(function (Vehicle $record): string {
                        return match ($this->resolveNextMaintenance($record)['status']) {
                            'overdue' => 'Overdue',
                            'soon'    => 'Due soon',
                            default   => 'OK',
                        };
                    })
                    ->color(function (Vehicle $record): string {
                        return match ($this->resolveNextMaintenance($record)['status']) {
                            'overdue' => 'danger',
                            'soon'    => 'warning',
                            default   => 'gray',
                        };
                    }),

                // What needs doing next (COF / Rego / Service)
                Tables\Columns\TextColumn::make('next_job')
                    ->label('Job')
                    ->badge()
                    ->alignCenter()
                    ->getStateUsing(fn (Vehicle $record): string => $this->resolveNextMaintenance($record)['job'])
                    ->color(function (Vehicle $record): string {
                        $next = $this->resolveNextMaintenance($record);

                        return match ($next['status']) {
                            'overdue' => 'danger',
                            'soon'    => 'warning',
                            default   => 'gray',
                        };
                    }),

                // The due date / km for that job
                Tables\Columns\TextColumn::make('next_due')
                    ->label('Due')
                    ->getStateUsing(fn (Vehicle $record): ?string => $this->resolveNextMaintenance($record)['due_label'])
                    ->description(fn (Vehicle $record): ?string => $this->resolveNextMaintenance($record)['human_delta'])
                    ->color(function (Vehicle $record): ?string {
                        $next = $this->resolveNextMaintenance($record);

                        if ($next['type'] === 'service') {
                            $kmRemaining = $next['km_remaining'];

                            if ($kmRemaining !== null && $kmRemaining <= 1000) {
                                return 'danger'; // <= 1000 km
                            }

                            if ($next['status'] === 'soon') {
                                return 'warning';
                            }

                            return null;
                        }

                        if (in_array($next['type'], ['cof', 'rego'], true)) {
                            $daysUntil = $next['days_until'];

                            if ($daysUntil !== null && $daysUntil <= 7) {
                                return 'danger'; // within 7 days
                            }

                            if ($next['status'] === 'soon') {
                                return 'warning'; // within 30 days
                            }

                            return null;
                        }

                        return null;
                    }),

                // Rego due – detail column, only highlighted when near
                Tables\Columns\TextColumn::make('rego_due_at')
                    ->label('Rego due')
                    ->date('Y-m-d')
                    ->sortable()
                    ->color(function (Vehicle $record): ?string {
                        if (! $record->rego_due_at instanceof Carbon) {
                            return null;
                        }

                        $today     = Carbon::today();
                        $daysUntil = (int) $today->diffInDays($record->rego_due_at, false);

                        if ($daysUntil <= 7) {
                            return 'danger'; // red if within 7 days / overdue
                        }

                        if ($daysUntil <= 30) {
                            return 'warning'; // orange if within 30 days
                        }

                        return null; // neutral otherwise
                    }),

                // Service due with "km to go"
                Tables\Columns\TextColumn::make('service_due_km')
                    ->label('Service due (km)')
                    ->alignRight()
                    ->formatStateUsing(function (?int $state): ?string {
                        if ($state === null) {
                            return null;
                        }

                        return number_format($state);
                    })
                    ->description(function (Vehicle $record): ?string {
                        if (! $record->service_due_km || ! $record->odometer_km) {
                            return null;
                        }

                        $remaining = (int) ($record->service_due_km - $record->odometer_km);

                        if ($remaining <= 0) {
                            return 'Due now';
                        }

                        if ($remaining <= 1000) {
                            return number_format($remaining) . ' km (soon)';
                        }

                        return number_format($remaining) . ' km to go';
                    })
                    ->color(function (Vehicle $record): ?string {
                        if (! $record->service_due_km || ! $record->odometer_km) {
                            return null;
                        }

                        $remaining = (int) ($record->service_due_km - $record->odometer_km);

                        if ($remaining <= 0 || $remaining <= 1000) {
                            return 'danger'; // overdue or within 1,000 km
                        }

                        if ($remaining <= 2000) {
                            return 'warning'; // getting close
                        }

                        return null; // neutral otherwise
                    })
                    ->toggleable(),
            ])
            ->actions([
                Tables\Actions\Action::make('logMaintenance')
                    ->label(function (Vehicle $record): string {
                        $next = $this->resolveNextMaintenance($record);

                        return $next['job'] === 'OK'
                            ? 'Log maintenance'
                            : 'Log ' . $next['job'];
                    })
                    ->icon('heroicon-o-wrench')
                    ->button()
                    ->color('primary')
                    ->url(fn (Vehicle $record): string => $this->buildMaintenanceCreateUrl($record))
                    ->openUrlInNewTab(),
            ])
            ->recordUrl(fn (Vehicle $record) => VehicleResource::getUrl('edit', [
                'record' => $record,
            ]))
            ->recordAction(null)
            ->paginated(false);
    }

    /**
     * Build URL to create a new Vehicle Maintenance Log, with sensible defaults:
     * - vehicle_id
     * - type (cof|rego|service|ok)
     * - cof_due_at / rego_due_at = today + 6 months (for cof/rego types)
     */
    private function buildMaintenanceCreateUrl(Vehicle $vehicle): string
    {
        $next = $this->resolveNextMaintenance($vehicle);

        $type = $next['type'] ?? 'ok';

        $query = [
            'vehicle_id' => $vehicle->id,
            'type'       => $type,
        ];

        // COF/rego default due = 6 months from today
        if (in_array($type, ['cof', 'rego'], true)) {
            $future = now()->addMonthsNoOverflow(6)->toDateString();

            if ($type === 'cof') {
                $query['cof_due_at'] = $future;
            }

            if ($type === 'rego') {
                $query['rego_due_at'] = $future;
            }
        }

        $queryString = http_build_query($query);

        return VehicleMaintenanceLogResource::getUrl('create') . '?' . $queryString;
    }

    /**
     * Base query:
     * - Orders by urgency bucket first (overdue -> soon -> OK),
     *   then by how close the next action is (km / days),
     *   so the very first row is the next action you should take.
     * - Only the top 5 vehicles are returned.
     * - Eager-loads jobs (no status filter) so we can work out availability.
     */
    protected function getTableQuery(): Builder
    {
        return Vehicle::query()
            ->with(['jobs' => function ($q) {
                $q->whereNotNull('start_at')
                  ->whereNotNull('end_at')
                  ->orderBy('start_at');
            }])
            ->orderByRaw("
                -- 1) Urgency bucket: 0 = overdue, 1 = due soon, 2 = OK
                CASE
                    -- Overdue COF / Rego / Service
                    WHEN cof_due_at IS NOT NULL AND cof_due_at < CURDATE() THEN 0
                    WHEN rego_due_at IS NOT NULL AND rego_due_at < CURDATE() THEN 0
                    WHEN service_due_km IS NOT NULL
                         AND odometer_km IS NOT NULL
                         AND CAST(service_due_km AS SIGNED) <= CAST(odometer_km AS SIGNED) THEN 0

                    -- Due soon: within 30 days (COF/rego) or 2000 km (service)
                    WHEN service_due_km IS NOT NULL
                         AND odometer_km IS NOT NULL
                         AND (CAST(service_due_km AS SIGNED) - CAST(odometer_km AS SIGNED)) <= 2000 THEN 1
                    WHEN cof_due_at IS NOT NULL
                         AND cof_due_at BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1
                    WHEN rego_due_at IS NOT NULL
                         AND rego_due_at BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) THEN 1

                    -- Everything else
                    ELSE 2
                END ASC,
                -- 2) Within the same bucket, order by 'closeness':
                --    service km remaining (scaled) first, then COF/rego days
                CASE
                    WHEN service_due_km IS NOT NULL AND odometer_km IS NOT NULL
                        THEN GREATEST(
                            CAST(service_due_km AS SIGNED) - CAST(odometer_km AS SIGNED),
                            0
                        ) / 100
                    ELSE 999999
                END ASC,
                DATEDIFF(cof_due_at, CURDATE()) ASC,
                DATEDIFF(rego_due_at, CURDATE()) ASC
            ")
            ->limit(5);
    }

    /**
     * Decide which maintenance job is the next priority for this vehicle.
     *
     * Returns:
     * [
     *   'job'          => 'COF' | 'Rego' | 'Service' | 'OK',
     *   'type'         => 'cof' | 'rego' | 'service' | 'ok',
     *   'due_label'    => string|null,
     *   'status'       => 'overdue'|'soon'|'ok',
     *   'days_until'   => int|null,
     *   'km_remaining' => int|null,
     *   'human_delta'  => string|null,
     * ]
     */
    private function resolveNextMaintenance(Vehicle $vehicle): array
    {
        $today = Carbon::today();

        $cof   = $vehicle->cof_due_at instanceof Carbon ? $vehicle->cof_due_at : null;
        $rego  = $vehicle->rego_due_at instanceof Carbon ? $vehicle->rego_due_at : null;
        $svcKm = $vehicle->service_due_km;
        $odo   = $vehicle->odometer_km;

        $cofDays  = $cof  ? (int) $today->diffInDays($cof, false) : null;
        $regoDays = $rego ? (int) $today->diffInDays($rego, false) : null;
        $svcRem   = ($svcKm && $odo) ? (int) ($svcKm - $odo) : null;

        // Helper to format human-friendly delta for dates
        $formatDateDelta = function ($days): ?string {
            if ($days === null) {
                return null;
            }

            $days = (int) $days;

            if ($days < 0) {
                $d = abs($days);

                return $d === 1 ? '1 day ago' : "{$d} days ago";
            }

            if ($days === 0) {
                return 'today';
            }

            if ($days === 1) {
                return 'in 1 day';
            }

            return "in {$days} days";
        };

        // Helper to format human-friendly delta for kms
        $formatKmDelta = function ($kmRem): ?string {
            if ($kmRem === null) {
                return null;
            }

            $kmRem = (int) $kmRem;

            if ($kmRem <= 0) {
                return 'due now';
            }

            if ($kmRem <= 1000) {
                return number_format($kmRem) . ' km (soon)';
            }

            return number_format($kmRem) . ' km to go';
        };

        // 1. Overdue checks first (most important)
        if ($cof && $cofDays !== null && $cofDays < 0) {
            return [
                'job'          => 'COF',
                'type'         => 'cof',
                'due_label'    => $cof->format('Y-m-d'),
                'status'       => 'overdue',
                'days_until'   => $cofDays,
                'km_remaining' => null,
                'human_delta'  => $formatDateDelta($cofDays),
            ];
        }

        if ($rego && $regoDays !== null && $regoDays < 0) {
            return [
                'job'          => 'Rego',
                'type'         => 'rego',
                'due_label'    => $rego->format('Y-m-d'),
                'status'       => 'overdue',
                'days_until'   => $regoDays,
                'km_remaining' => null,
                'human_delta'  => $formatDateDelta($regoDays),
            ];
        }

        if ($svcRem !== null && $svcRem <= 0) {
            return [
                'job'          => 'Service',
                'type'         => 'service',
                'due_label'    => number_format($svcKm) . ' km (due now)',
                'status'       => 'overdue',
                'days_until'   => null,
                'km_remaining' => $svcRem,
                'human_delta'  => $formatKmDelta($svcRem),
            ];
        }

        // 2. Upcoming "soon" (within 30 days / 2000 km)
        $soonDays = 30;
        $soonKm   = 2000;

        $candidates = [];

        if ($cofDays !== null && $cofDays >= 0 && $cofDays <= $soonDays) {
            $candidates[] = [
                'job'          => 'COF',
                'type'         => 'cof',
                'due_label'    => $cof->format('Y-m-d'),
                'status'       => 'soon',
                'days_until'   => $cofDays,
                'km_remaining' => null,
                'sort_key'     => $cof->timestamp,
                'human_delta'  => $formatDateDelta($cofDays),
            ];
        }

        if ($regoDays !== null && $regoDays >= 0 && $regoDays <= $soonDays) {
            $candidates[] = [
                'job'          => 'Rego',
                'type'         => 'rego',
                'due_label'    => $rego->format('Y-m-d'),
                'status'       => 'soon',
                'days_until'   => $regoDays,
                'km_remaining' => null,
                'sort_key'     => $rego->timestamp,
                'human_delta'  => $formatDateDelta($regoDays),
            ];
        }

        if ($svcRem !== null && $svcRem > 0 && $svcRem <= $soonKm) {
            $candidates[] = [
                'job'          => 'Service',
                'type'         => 'service',
                'due_label'    => number_format($svcKm) . ' km',
                'status'       => 'soon',
                'days_until'   => null,
                'km_remaining' => $svcRem,
                'sort_key'     => (int) $svcKm,
                'human_delta'  => $formatKmDelta($svcRem),
            ];
        }

        if (! empty($candidates)) {
            usort($candidates, fn (array $a, array $b): int => $a['sort_key'] <=> $b['sort_key']);

            $next = $candidates[0];
            unset($next['sort_key']);

            return $next;
        }

        // 3. Nothing urgent – fall back to earliest of COF/rego/service if present
        if ($cof || $rego || $svcKm) {
            $fallback = [];

            if ($cof) {
                $fallback[] = [
                    'job'          => 'COF',
                    'type'         => 'cof',
                    'due_label'    => $cof->format('Y-m-d'),
                    'status'       => 'ok',
                    'days_until'   => $cofDays,
                    'km_remaining' => null,
                    'sort_key'     => $cof->timestamp,
                    'human_delta'  => $formatDateDelta($cofDays),
                ];
            }

            if ($rego) {
                $fallback[] = [
                    'job'          => 'Rego',
                    'type'         => 'rego',
                    'due_label'    => $rego->format('Y-m-d'),
                    'status'       => 'ok',
                    'days_until'   => $regoDays,
                    'km_remaining' => null,
                    'sort_key'     => $rego->timestamp,
                    'human_delta'  => $formatDateDelta($regoDays),
                ];
            }

            if ($svcKm) {
                $fallback[] = [
                    'job'          => 'Service',
                    'type'         => 'service',
                    'due_label'    => number_format($svcKm) . ' km',
                    'status'       => 'ok',
                    'days_until'   => null,
                    'km_remaining' => $svcRem,
                    'sort_key'     => (int) $svcKm,
                    'human_delta'  => $formatKmDelta($svcRem),
                ];
            }

            usort($fallback, fn (array $a, array $b): int => $a['sort_key'] <=> $b['sort_key']);

            $next = $fallback[0];
            unset($next['sort_key']);

            return $next;
        }

        // 4. Nothing configured
        return [
            'job'          => 'OK',
            'type'         => 'ok',
            'due_label'    => null,
            'status'       => 'ok',
            'days_until'   => null,
            'km_remaining' => null,
            'human_delta'  => null,
        ];
    }

    /**
     * Resolve current / next booking for this vehicle.
     */
    private function resolveNextBooking(Vehicle $vehicle): array
    {
        $now  = now();
        $jobs = $vehicle->jobs; // eager-loaded in getTableQuery()

        if (! $jobs || $jobs->isEmpty()) {
            return [
                'status'      => 'free',
                'label'       => 'No bookings',
                'description' => null,
            ];
        }

        // 1) Is it currently on hire?
        $current = $jobs->first(function ($job) use ($now) {
            if (! $job->start_at || ! $job->end_at) {
                return false;
            }

            return $job->start_at <= $now && $job->end_at >= $now;
        });

        if ($current) {
            $endDate = $current->end_at?->format('Y-m-d') ?? '';
            $days    = $current->end_at
                ? (int) $now->diffInDays($current->end_at, false)
                : null;

            $desc = $days !== null && $days >= 0
                ? ($days === 0 ? 'Ends today' : "Ends in {$days} day" . ($days === 1 ? '' : 's'))
                : null;

            return [
                'status'      => 'current',
                'label'       => "On hire until {$endDate}",
                'description' => $desc,
            ];
        }

        // 2) Next upcoming booking
        $upcoming = $jobs->first(function ($job) use ($now) {
            return $job->start_at && $job->start_at > $now;
        });

        if ($upcoming) {
            $startDate = $upcoming->start_at?->format('Y-m-d') ?? '';
            $days      = (int) $now->diffInDays($upcoming->start_at, false);

            $desc = $days === 0
                ? 'Starts today'
                : "Starts in {$days} day" . ($days === 1 ? '' : 's');

            // Treat <= 3 days as "soon", anything further out as "free" (good window)
            $status = $days <= 3 ? 'soon' : 'free';

            return [
                'status'      => $status,
                'label'       => "Starts {$startDate}",
                'description' => $desc,
            ];
        }

        // 3) Jobs exist but none current/upcoming (edge case)
        return [
            'status'      => 'free',
            'label'       => 'No active bookings',
            'description' => null,
        ];
    }
}
