<?php

declare(strict_types=1);

namespace App\Filament\Widgets;

use App\Models\Job;
use App\Models\Payment;
use Carbon\CarbonInterface;
use Filament\Tables;
use Filament\Widgets\TableWidget as Widget;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Schema;

class UpcomingPickupsTable extends Widget
{
    protected static ?string $heading = 'Upcoming Pickups (7 days)';
    protected int|array|string $columnSpan = 'full';
    protected static ?string $pollingInterval = '15s';

    /** Pick the pickup datetime column: start_at → pickup_at → from → created_at */
    protected function pickupColumn(): string
    {
        foreach (['start_at', 'pickup_at', 'from', 'created_at'] as $col) {
            if (Schema::hasColumn('jobs', $col)) {
                return $col;
            }
        }
        return 'created_at';
    }

    /** Pick a customer field if present */
    protected function customerColumns(): array
    {
        foreach (['customer_name', 'c_name', 'c_driver_name', 'name'] as $col) {
            if (Schema::hasColumn('jobs', $col)) {
                return [$col];
            }
        }
        return [];
    }

    protected function getTableQuery(): Builder
    {
        $pickupCol = $this->pickupColumn();
        $from = now();
        $to   = now()->addDays(7);

        // Sum of PAID payments per job (in cents)
        $paidSub = Payment::query()
            ->selectRaw('job_id, COALESCE(SUM(amount_cents), 0) AS paid_cents')
            ->whereIn('status', ['succeeded', 'paid', 'captured'])
            ->groupBy('job_id');

        // Build charge_cents expression dynamically depending on schema
        if (Schema::hasColumn('jobs', 'charge_amount_cents')) {
            $chargeExpr = 'COALESCE(jobs.charge_amount_cents, 0)';
        } elseif (Schema::hasColumn('jobs', 'charge_amount')) {
            $chargeExpr = 'CAST(ROUND(COALESCE(jobs.charge_amount, 0) * 100) AS SIGNED)';
        } elseif (Schema::hasColumn('jobs', 'total_price')) {
            $chargeExpr = 'CAST(ROUND(COALESCE(jobs.total_price, 0) * 100) AS SIGNED)';
        } else {
            $chargeExpr = '0';
        }

        return Job::query()
            ->leftJoinSub($paidSub, 'p', 'p.job_id', '=', 'jobs.id')
            ->whereBetween("jobs.$pickupCol", [$from, $to])
            ->select('jobs.*')
            ->selectRaw("$chargeExpr AS charge_cents")
            ->selectRaw('COALESCE(p.paid_cents, 0) AS paid_cents')
            ->selectRaw("GREATEST(COALESCE(($chargeExpr),0) - COALESCE(p.paid_cents,0), 0) AS due_cents")
            ->orderBy("jobs.$pickupCol");
    }

    protected function getTableColumns(): array
    {
        $pickupCol   = $this->pickupColumn();
        $nameColumns = $this->customerColumns();
        $customerCol = $nameColumns[0] ?? null;

        $columns = [
            // 🚩 Flag only when there is an unpaid balance
            Tables\Columns\IconColumn::make('flag')
                ->label('')
                ->icon('heroicon-m-flag')
                ->color('danger')
                ->tooltip('Unpaid balance')
                ->visible(fn ($record): bool => (int) ($record->due_cents ?? 0) > 0),

            // External reference as Job (clickable)
            Tables\Columns\TextColumn::make('external_reference')
                ->label('Job')
                ->formatStateUsing(fn ($state) => $state ?: '—')
                ->url(fn ($record) => url("/admin/jobs/{$record->id}"))
                ->openUrlInNewTab()
                ->sortable()
                ->searchable(),

            // Customer name (with graceful fallbacks)
            Tables\Columns\TextColumn::make($customerCol ?? 'customer_fallback')
                ->label('Customer')
                ->formatStateUsing(function ($record) use ($customerCol): string {
                    if ($customerCol && !empty($record->{$customerCol})) {
                        return (string) $record->{$customerCol};
                    }
                    foreach (['c_name', 'c_driver_name', 'customer_name', 'name'] as $alt) {
                        if (!empty($record->{$alt})) {
                            return (string) $record->{$alt};
                        }
                    }
                    return '—';
                })
                ->searchable(),

            // Vehicle (if relationship/column exists)
            Tables\Columns\TextColumn::make('vehicle.name')
                ->label('Vehicle')
                ->formatStateUsing(fn ($state) => $state ?: '—'),
        ];

        // Pickup datetime
        $columns[] = Tables\Columns\TextColumn::make($pickupCol)
            ->label('Pickup')
            ->formatStateUsing(function ($state): string {
                if (!$state) {
                    return '—';
                }
                try {
                    $dt = $state instanceof CarbonInterface ? $state : Carbon::parse($state);
                    return $dt->isoFormat('ddd D MMM, HH:mm');
                } catch (\Throwable $e) {
                    return (string) $state;
                }
            })
            ->sortable();

        // Amount due (computed)
        $columns[] = Tables\Columns\TextColumn::make('due_cents')
            ->label('Amount Due')
            ->formatStateUsing(fn ($state) => 'NZD ' . number_format(((int) $state) / 100, 2))
            ->color(fn ($state) => ((int) $state) > 0 ? 'danger' : null)
            ->sortable();

        return $columns;
    }
}
