<?php

declare(strict_types=1);

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\Deposit;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Stripe\StripeClient;
use Throwable;

class DepositCaptureController extends Controller
{
    protected function stripe(): StripeClient
    {
        return new StripeClient((string) config('services.stripe.secret'));
    }

    /**
     * Create an authorization HOLD for a Job using capture_method=manual.
     * Expects: amount_cents (int), payment_method_id (string from Stripe.js), optional metadata array.
     */
    public function createHold(Request $request, Job $job)
    {
        $data = $request->validate([
            'amount_cents'      => ['required','integer','min:1'],
            'payment_method_id' => ['required','string'],
            'metadata'          => ['sometimes','array'],
        ]);

        $amount   = (int) $data['amount_cents'];
        $pm       = (string) $data['payment_method_id'];
        $currency = strtolower($job->currency ?? optional($job->flow)->currency ?? 'nzd');

        try {
            $pi = $this->stripe()->paymentIntents->create([
                'amount'               => $amount,
                'currency'             => $currency,
                'payment_method_types' => ['card'],
                'payment_method'       => $pm,
                'capture_method'       => 'manual',   // <-- key for holds
                'confirm'              => true,
                'metadata'             => array_merge([
                    'job_id'   => (string) $job->id,
                    'source'   => 'admin',
                    'type'     => 'deposit_hold',
                ], $data['metadata'] ?? []),
            ]);

            // Store a Deposit row representing the authorization
            DB::transaction(function () use ($job, $pi, $amount, $currency) {
                Deposit::create([
                    'job_id'                 => $job->id,
                    'status'                 => 'authorized',   // authorized | captured | canceled | released
                    'authorized_cents'       => $amount,
                    'captured_cents'         => 0,
                    'currency'               => strtoupper($currency),
                    'stripe_payment_intent'  => $pi->id,
                    'stripe_charge_id'       => null,           // known after capture
                ]);
            });

            return back()->with('success', "Hold created on PI {$pi->id} for {$amount} cents.");
        } catch (Throwable $e) {
            Log::error('Create hold failed', ['job_id' => $job->id, 'err' => $e->getMessage()]);
            return back()->with('error', 'Failed to create hold: '.$e->getMessage());
        }
    }

    /**
     * Capture a hold (full or partial).
     * Expects: amount_cents (int, optional). If omitted, Stripe captures remaining capturable.
     */
    public function capture(Request $request, Deposit $deposit)
    {
        $data = $request->validate([
            'amount_cents' => ['nullable','integer','min:1'],
        ]);

        if ($deposit->status !== 'authorized') {
            return back()->with('error', "Deposit is not in 'authorized' state.");
        }

        $piId   = (string) $deposit->stripe_payment_intent;
        $amount = $data['amount_cents'] ?? $deposit->remainingHoldCents();

        if ($amount < 1) {
            return back()->with('error', 'Nothing left to capture.');
        }
        if ($amount > $deposit->remainingHoldCents()) {
            return back()->with('error', 'Amount exceeds remaining authorized amount.');
        }

        try {
            $params = [];
            // Only pass amount_to_capture if partial; Stripe will capture max if omitted
            if ($amount !== $deposit->remainingHoldCents()) {
                $params['amount_to_capture'] = $amount;
            }

            $pi = $this->stripe()->paymentIntents->capture($piId, $params);

            // Latest charge id (for bookkeeping). PI may have multiple charges but for cards there’s one latest.
            $latestChargeId = is_string($pi->latest_charge ?? null) ? $pi->latest_charge : null;

            DB::transaction(function () use ($deposit, $amount, $latestChargeId, $pi) {
                // Upsert a Payment row to reflect captured funds (optional but recommended)
                Payment::firstOrCreate(
                    ['stripe_payment_intent' => $pi->id],
                    [
                        'job_id'        => $deposit->job_id,
                        'amount_cents'  => 0,
                        'currency'      => $deposit->currency ?? 'NZD',
                        'status'        => 'succeeded',
                    ]
                )->increment('amount_cents', $amount);

                // Update deposit
                $deposit->captured_cents += $amount;
                if ($latestChargeId) {
                    $deposit->stripe_charge_id = $latestChargeId;
                }
                $deposit->status = $deposit->remainingHoldCents() > 0 ? 'authorized' : 'captured';
                $deposit->save();
            });

            return back()->with('success', "Captured {$amount} cents on PI {$piId}.");
        } catch (Throwable $e) {
            Log::error('Capture hold failed', ['deposit_id' => $deposit->id, 'pi' => $piId, 'err' => $e->getMessage()]);
            return back()->with('error', 'Failed to capture: '.$e->getMessage());
        }
    }

    /**
     * Cancel (release) the authorization.
     */
    public function cancel(Request $request, Deposit $deposit)
    {
        if ($deposit->status !== 'authorized') {
            return back()->with('error', "Deposit is not in 'authorized' state.");
        }

        $piId = (string) $deposit->stripe_payment_intent;

        try {
            $this->stripe()->paymentIntents->cancel($piId, [
                'cancellation_reason' => 'requested_by_customer',
            ]);

            $deposit->status           = 'canceled';
            $deposit->authorized_cents = 0;
            $deposit->save();

            return back()->with('success', "Hold canceled for PI {$piId}.");
        } catch (Throwable $e) {
            Log::error('Cancel hold failed', ['deposit_id' => $deposit->id, 'pi' => $piId, 'err' => $e->getMessage()]);
            return back()->with('error', 'Failed to cancel: '.$e->getMessage());
        }
    }
}
