<?php 
// app/Http/Controllers/Admin/OneOffPaymentController.php
class OneOffPaymentController extends Controller {
    public function store(Job $job, Request $r) {
        $data = $r->validate([
            'amount_cents' => 'required|integer|min:100',
            'currency'     => 'nullable|string|size:3',
            'reason'       => 'nullable|string|max:2000',
            'mode'         => 'required|in:request_link,off_session',
            'attachment'   => 'nullable|file|max:8192', // 8MB
        ]);

        $p = DB::transaction(function() use ($job,$data,$r) {
            $p = Payment::create([
                'job_id'        => $job->id,
                'amount_cents'  => $data['amount_cents'],
                'currency'      => $data['currency'] ?? 'NZD',
                'status'        => 'pending',
                'type'          => 'one_off',
                'charge_mode'   => $data['mode'],
                'reason'        => $data['reason'] ?? null,
                'public_token'  => (string) Str::uuid(),
                'gateway'       => 'stripe',
            ]);

            if ($r->hasFile('attachment')) {
                $file = $r->file('attachment');
                $path = $file->store("payments/{$p->id}", ['disk' => config('filesystems.default')]);
                $p->attachments()->create([
                    'disk'          => config('filesystems.default'),
                    'path'          => $path,
                    'original_name' => $file->getClientOriginalName(),
                    'mime'          => $file->getClientMimeType(),
                    'size_bytes'    => $file->getSize(),
                ]);
            }

            event(new \App\Events\JobEvent($job, 'one_off_payment_created', ['payment_id'=>$p->id]));
            return $p;
        });

        return match ($data['mode']) {
            'request_link' => $this->initCheckoutAndEmail($p),
            'off_session'  => $this->attemptOffSessionCharge($p),
        };
    }
}
