<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\Vehicle;
use App\Models\VehicleMaintenanceLog;
use Carbon\Carbon;
use Illuminate\Http\Request;

class FleetCalendarController extends Controller
{
    /**
     * Vehicles as resources for the fleet calendar.
     */
    public function resources()
    {
        return Vehicle::query()
            ->orderBy('name')
            ->get()
            ->map(fn (Vehicle $vehicle) => [
                'id'    => "vehicle-{$vehicle->id}",
                'title' => $vehicle->display_name, // e.g. "QAA322 · Jimny 02"
            ])
            ->values();
    }

    /**
     * Jobs + maintenance events for the fleet calendar.
     */
    public function events(Request $request)
    {
        $start = Carbon::parse($request->query('start'));
        $end   = Carbon::parse($request->query('end'));

        //
        // ─────────── JOB EVENTS (hires) ───────────
        //

        $jobs = Job::query()
            // ->forCalendar()
            ->whereNotNull('start_at')
            ->whereNotNull('end_at')
            ->where(function ($q) use ($start, $end) {
                $q->whereBetween('start_at', [$start, $end])
                  ->orWhereBetween('end_at', [$start, $end])
                  ->orWhere(function ($q2) use ($start, $end) {
                      $q2->where('start_at', '<=', $start)
                         ->where('end_at', '>=', $end);
                  });
            })
            ->with('vehicle')
            ->get()
            ->map(function (Job $job) {
                return [
                    'id'         => "job-{$job->ref_id}",
                    'title'      => '', // label built in JS
                    'type'       => 'hire',
                    'resourceId' => "vehicle-{$job->vehicle_id}",
                    'start'      => optional($job->start_at)->toIso8601String(),
                    'end'        => optional($job->end_at)->toIso8601String(),

                    // extra info for JS label
                    'job_ref'    => $job->ref_id,
                    'customer'   => $job->customer_name,

                    'url'        => route('filament.admin.resources.jobs.view', ['record' => $job]),
                ];
            });

        //
        // ─────────── MAINTENANCE EVENTS (1-day soft blocks) ───────────
        //

        $maintenance = VehicleMaintenanceLog::query()
            ->whereNotNull('performed_at')
            ->whereBetween('performed_at', [
                $start->toDateString(),
                $end->toDateString(),
            ])
            ->with('vehicle')
            ->get()
            ->map(function (VehicleMaintenanceLog $log) {
                $start = Carbon::parse($log->performed_at)->startOfDay();
                $end   = (clone $start)->addDay();

                return [
                    'id'         => "maint-{$log->id}",
                    'title'      => '', // label built in JS
                    'type'       => 'maintenance',
                    'resourceId' => "vehicle-{$log->vehicle_id}",
                    'start'      => $start->toIso8601String(),
                    'end'        => $end->toIso8601String(),

                    'job_ref'    => null,
                    'customer'   => null,

                    'url'        => null,
                ];
            });

        return $jobs->concat($maintenance)->values();
    }
}
