<?php

declare(strict_types=1);

namespace App\Http\Controllers;

use App\Models\Job;
use App\Models\Vehicle;
use App\Models\VehicleMaintenanceLog;
use Carbon\Carbon;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class FleetCalendarController extends Controller
{
    /**
     * Timezone used for the fleet calendar.
     */
    private const TZ = 'Pacific/Auckland';

    /**
     * Return the list of vehicles as FullCalendar "resources".
     *
     * GET /fleet-calendar/resources
     */
    public function resources(Request $request): JsonResponse
    {
        $vehicles = Vehicle::query()
            ->orderBy('registration')
            ->orderBy('name')
            ->get();

        $resources = $vehicles->map(function (Vehicle $vehicle) {
            // Build a nice label like "QAA322 · Jimny 02"
            $parts = [];

            if ($vehicle->registration ?? null) {
                $parts[] = $vehicle->registration;
            }

            if ($vehicle->name ?? null) {
                $parts[] = $vehicle->name;
            }

            if (empty($parts) && ($vehicle->display_name ?? null)) {
                $parts[] = $vehicle->display_name;
            }

            $title = implode(' · ', array_filter($parts));

            if ($title === '') {
                $title = 'Vehicle #' . $vehicle->id;
            }

            return [
                'id'    => $vehicle->id,
                'title' => $title,
            ];
        });

        return response()->json($resources->values());
    }

    /**
     * Return hires + maintenance entries as FullCalendar events.
     *
     * GET /fleet-calendar/events
     */
    public function events(Request $request): JsonResponse
{
    $tz = self::TZ;

    // Use the range from FullCalendar if provided, otherwise a sensible default
    $rangeStart = $request->query('start')
        ? Carbon::parse($request->query('start'))->setTimezone($tz)
        : now($tz)->subMonths(2);

    $rangeEnd = $request->query('end')
        ? Carbon::parse($request->query('end'))->setTimezone($tz)
        : now($tz)->addMonths(2);

    $events = [];

    $jobs = Job::query()
        ->whereNotNull('vehicle_id')
        ->get();

    foreach ($jobs as $job) {
        // Try multiple possible field names for start/end datetime.
        // This way it still works regardless of whether you called them
        // start_at / pickup_at / from, etc.
        $rawStart = $job->start_at
            ?? $job->pickup_at
            ?? $job->from
            ?? null;

        $rawEnd = $job->end_at
            ?? $job->return_at
            ?? $job->to
            ?? null;

        // If you also store separate date+time columns, you can combine them here:
        if (! $rawStart && ($job->start_date ?? null)) {
            $rawStart = trim(($job->start_date ?? '') . ' ' . ($job->start_time ?? '00:00:00'));
        }

        if (! $rawEnd && ($job->end_date ?? null)) {
            $rawEnd = trim(($job->end_date ?? '') . ' ' . ($job->end_time ?? '00:00:00'));
        }

        $start = $this->normalizeDateTime($rawStart, $tz);
        $end   = $this->normalizeDateTime($rawEnd, $tz);

        if (! $start && ! $end) {
            continue;
        }

        // Skip events totally outside the requested range
        if ($start && $end) {
            $startC = Carbon::parse($start);
            $endC   = Carbon::parse($end);

            if ($endC->lt($rangeStart) || $startC->gt($rangeEnd)) {
                continue;
            }
        }

        $events[] = [
            'id'         => 'job-' . $job->id,
            'title'      => $job->customer_name ?? 'Hire',
            'start'      => $start,
            'end'        => $end,
            'resourceId' => $job->vehicle_id,
            'extendedProps' => [
                'type'     => 'hire',
                'customer' => $job->customer_name ?? null,
                'job_ref'  => $job->booking_reference ?? $job->id,
                'url'      => route('filament.admin.resources.jobs.edit', $job),
            ],
        ];
    }

    // (Maintenance block stays as I wrote earlier – that won’t affect the hires.)

    return response()->json(array_values($events));
}


    /**
     * Resolve a Job start/end datetime into an ISO8601 string in the given tz.
     *
     * Priority:
     *   1. start_at / end_at (Carbon or string)
     *   2. start_date + start_time / end_date + end_time
     */
    protected function jobDateTime(Job $job, string $prefix, string $tz): ?string
    {
        $atAttr = $prefix . '_at';
        $dateAttr = $prefix . '_date';
        $timeAttr = $prefix . '_time';

        // 1. If we have start_at / end_at
        if ($job->{$atAttr} ?? null) {
            return $this->normalizeDateTime($job->{$atAttr}, $tz);
        }

        // 2. Fallback to date + time
        $date = $job->{$dateAttr} ?? null;
        if (! $date) {
            return null;
        }

        $time = $job->{$timeAttr} ?? '00:00:00';

        try {
            $dt = Carbon::parse(trim($date . ' ' . $time), $tz);

            return $dt->setTimezone($tz)->toIso8601String();
        } catch (\Throwable $e) {
            return null;
        }
    }

    /**
     * Normalise any datetime-ish value into an ISO8601 string in the given tz.
     *
     * Accepts:
     *   - Carbon instance
     *   - string
     *   - null
     */
    protected function normalizeDateTime(mixed $value, string $tz): ?string
    {
        if ($value === null) {
            return null;
        }

        if ($value instanceof Carbon) {
            return $value->clone()->setTimezone($tz)->toIso8601String();
        }

        try {
            $dt = Carbon::parse((string) $value, $tz);

            return $dt->setTimezone($tz)->toIso8601String();
        } catch (\Throwable $e) {
            return null;
        }
    }
}
