<?php

declare(strict_types=1);

namespace App\Http\Controllers\Payments;

use App\Http\Controllers\Controller;
use App\Models\PaymentLink;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Stripe\StripeClient;
use Illuminate\Support\Carbon;

class PublicPayController extends Controller
{
    /** Show the public pay page */
    /** Show the public pay page */
public function show(string $token)
{
    $link = PaymentLink::with('payment')->where('public_token', $token)->firstOrFail();
    abort_if(optional($link->expires_at)->isPast(), 410);

    $p = $link->payment;

    // If already paid, bounce to thanks immediately
    if (in_array($p->status, ['succeeded', 'paid', 'completed'], true)) {
        return redirect()
            ->route('payments.public.done', ['token' => $token])
            ->with('status', 'already_paid');
    }

    $amountCents = (int) (
        $p->amount_cents
        ?: (isset($p->amount) ? round(((float) $p->amount) * 100) : 0)
    );

    $isBondHold = in_array($p->type ?? '', ['bond_hold', 'hold'], true)
               || (($p->purpose ?? null) === 'hold');

    // ---- Resolve booking dates (many possible columns) ----
    $job = $p->job ?? null;

    $startRaw = $job->start_at
        ?? $job->pickup_at
        ?? $job->start_date
        ?? data_get($job, 'meta.start_at')
        ?? data_get($p,   'meta.start_at');

    $endRaw = $job->end_at
        ?? $job->return_at
        ?? $job->end_date
        ?? data_get($job, 'meta.end_at')
        ?? data_get($p,   'meta.end_at');

    $tz = $job->timezone ?? config('app.timezone', 'Pacific/Auckland');

    $startDate = $startRaw ? Carbon::parse($startRaw)->timezone($tz)->format('D d M, H:i') : null;
    $endDate   = $endRaw   ? Carbon::parse($endRaw)->timezone($tz)->format('D d M, H:i')   : null;

    return view('pay.show', [
        'token'       => $token,
        'amountCents' => $amountCents,
        'currency'    => $p->currency ?? 'NZD',
        'reference'   => $p->reference ?? $p->job?->external_reference,
        'stripePk'    => config('services.stripe.key') ?: config('payments.drivers.stripe.publishable'),
        'provider'    => $p->provider ?: config('payments.default', 'stripe'),
        'isBondHold'  => $isBondHold,
        'startDate'   => $startDate,
        'endDate'     => $endDate,
    ]);
}


    /**
     * Create a driver-agnostic session.
     * Stripe path implements:
     *  - Bond/hold: capture_method=manual + set up off-session auth (OSA)
     *  - Normal payment: set up OSA only if none exists for this customer yet
     */
    public function session(Request $request)
{
    $data = $request->validate([
        'token'         => ['required', 'string'],
        // optional extras if your form sends them (used for receipts / customer create)
        'receipt_email' => ['nullable', 'email'],
        'payer_name'    => ['nullable', 'string', 'max:255'],
    ]);

    $link = PaymentLink::with(['payment', 'payment.job', 'payment.job.customer'])
        ->where('public_token', $data['token'])->firstOrFail();

    abort_if(optional($link->expires_at)->isPast(), 410);

    $p = $link->payment;

    // Choose provider (extend later for Windcave, etc.)
    $provider = $p->provider ?: config('payments.default', 'stripe');
    if ($provider !== 'stripe') {
        return response()->json(['error' => 'Unsupported provider: ' . $provider], 400);
    }

    // ---------- Stripe path ----------
    $stripe = new StripeClient(
        config('services.stripe.secret') ?: config('payments.drivers.stripe.secret')
    );

    $isBondHold = in_array($p->type ?? '', ['bond_hold', 'hold'], true)
               || (($p->purpose ?? null) === 'hold');

    $amountCents = (int) (
        $p->amount_cents
        ?: (isset($p->amount) ? round(((float) $p->amount) * 100) : 0)
    );
    $currency = strtolower($p->currency ?? 'nzd');
    $desc     = $p->reference ?: ('Payment ' . $link->public_token);

    // Resolve (or create) a Stripe customer (cus_…). If not found, try to create from any known email.
    $stripeCustomerId = $this->resolveStripeCustomerId($stripe, $p);

    $possibleEmail = $data['receipt_email']
        ?? $p->customer_email
        ?? $p->job?->customer?->email
        ?? $p->email
        ?? null;

    if (!$stripeCustomerId && $possibleEmail) {
        try {
            $created = $stripe->customers->create([
                'email' => $possibleEmail,
                'name'  => trim($data['payer_name'] ?? $p->customer_name ?? $p->job?->customer?->name ?? $p->name ?? ''),
                'metadata' => [
                    'source'     => 'holds/public-pay',
                    'payment_id' => (string) $p->id,
                ],
            ]);
            $stripeCustomerId = $created->id;

            // Persist to your Customer model if you have a column for it
            if ($p->job?->customer && Schema::hasColumn($p->job->customer->getTable(), 'stripe_customer_id')) {
                $p->job->customer->stripe_customer_id = $stripeCustomerId;
                $p->job->customer->save();
            }
        } catch (\Throwable $e) {
            Log::info('Stripe customer create failed: ' . $e->getMessage());
        }
    }

    // Do we already have an off-session auth saved for this customer?
    $hasOSA = $this->hasOffSessionAuth($stripe, $stripeCustomerId);

    // Work out which column (if any) holds the PI id already
    $existingPiId = $this->firstExistingColumnValue($p, ['provider_id', 'gateway_id']);
    $pi = null;

    if ($existingPiId) {
        try {
            $pi = $stripe->paymentIntents->retrieve($existingPiId);

            // Mirror Stripe status (temporary, until webhook)
            if ($this->hasColumns('payments', ['status'])) {
                $p->status = $pi->status ?? $p->status;
                $p->save();
            }

            // If we need a hold but existing PI isn't manual capture, recreate
            if ($isBondHold && (($pi->capture_method ?? null) !== 'manual')) {
                $this->safeCancel($stripe, $existingPiId);
                $pi = null; $existingPiId = null;
            } else {
                // Reuse when still payable
                $reusable = in_array($pi->status, [
                    'requires_payment_method', 'requires_confirmation', 'requires_action', 'processing',
                ], true);

                if (!$reusable) {
                    $pi = null; // force creation of a new PI
                }
            }
        } catch (\Throwable $e) {
            Log::warning('Failed to retrieve existing PI: ' . $e->getMessage(), ['pi' => $existingPiId]);
            $pi = null; // fall through to create
        }
    }

    if (!$pi) {
        $params = [
            'amount'      => $amountCents,
            'currency'    => $currency,
            'description' => $desc,
            'metadata'    => [
                'payment_id'      => (string) $p->id,
                'payment_link_id' => (string) $link->id,
            ],
            // Use APM; no explicit payment_method_types
            'automatic_payment_methods' => ['enabled' => true, 'allow_redirects' => 'never'],
        ];

        // Attach customer if we have one so Stripe can save the card for OSA.
        if ($stripeCustomerId) {
            $params['customer'] = $stripeCustomerId;
        } elseif ($possibleEmail) {
            // If no customer, at least give Stripe an email for receipts.
            $params['receipt_email'] = $possibleEmail;
        }

        if ($isBondHold) {
            $params['capture_method'] = 'manual';
            $params['payment_method_options'] = [
                'card' => ['setup_future_usage' => 'off_session'],
            ];
            $params['metadata']['is_bond_hold'] = '1';
        } else {
            if (!$hasOSA) {
                $params['payment_method_options'] = [
                    'card' => ['setup_future_usage' => 'off_session'],
                ];
                $params['metadata']['set_up_osa'] = '1';
            }
        }

        // Idempotency: create one PI per payment (versioned by shape)
        $idemKey = sprintf(
            'pi_pay_v2_%d_%s_%d%s',
            $p->id,
            $isBondHold ? 'hold' : 'pay',
            $amountCents,
            !empty($params['capture_method']) ? '_'.$params['capture_method'] : ''
        );

        try {
            $pi = $stripe->paymentIntents->create($params, ['idempotency_key' => $idemKey]);
        } catch (\Stripe\Exception\InvalidRequestException $e) {
            if (str_contains($e->getMessage(), 'Keys for idempotent requests')) {
                $pi = $stripe->paymentIntents->create($params, ['idempotency_key' => $idemKey . '_r1']);
            } else {
                throw $e;
            }
        }

        // Persist identifiers ONLY to columns that exist
        $save = false;

        if ($this->hasColumns('payments', ['provider', 'provider_id'])) {
            $p->provider    = 'stripe';
            $p->provider_id = $pi->id;
            $save = true;
        } elseif ($this->hasColumns('payments', ['gateway', 'gateway_id'])) {
            $p->gateway    = 'stripe';
            $p->gateway_id = $pi->id;
            $save = true;
        }

        // Mirror Stripe status (temporary until webhook)
        if ($this->hasColumns('payments', ['status'])) {
            $p->status = $pi->status ?? $p->status;
            $save = true;
        }

        if ($save) {
            $p->save();
        } else {
            Log::warning('PublicPayController: No suitable columns to persist gateway ids/status on payments table.');
        }
    }

    // If we reused an existing PI, mirror status as well (in case not mirrored above)
    if ($pi && $this->hasColumns('payments', ['status'])) {
        try {
            if (($p->status ?? null) !== ($pi->status ?? null)) {
                $p->status = $pi->status ?? $p->status;
                $p->save();
            }
        } catch (\Throwable $e) {
            Log::info('Status mirror after reuse failed: ' . $e->getMessage());
        }
    }

    return response()->json([
        'driver'       => 'stripe',
        'clientSecret' => $pi->client_secret,
    ]);
}

    /** "Thanks" page, with a one-off Stripe verification to avoid repeat payments. */
    public function done(string $token)
    {
        $link = PaymentLink::with('payment')->where('public_token', $token)->first();

        if ($link && $link->payment) {
            $p = $link->payment;

            // If not already marked paid/authorized, verify with Stripe once on return
            if (!in_array($p->status, ['succeeded', 'paid', 'completed', 'authorized'], true)) {
                $stripeSecret = config('services.stripe.secret') ?: config('payments.drivers.stripe.secret');

                // Confirm we're dealing with Stripe and we have a PI id somewhere
                $isStripe = (
                    (Schema::hasColumn($p->getTable(), 'provider') && $p->provider === 'stripe') ||
                    (Schema::hasColumn($p->getTable(), 'gateway')  && $p->gateway  === 'stripe')
                );

                $piId = $this->firstExistingColumnValue($p, ['provider_id', 'gateway_id']);

                if ($stripeSecret && $isStripe && $piId) {
                    try {
                        $stripe = new StripeClient($stripeSecret);
                        $pi     = $stripe->paymentIntents->retrieve($piId);

                        // Map PI status → our Payment status
                        if ($pi && in_array($pi->status, ['succeeded', 'requires_capture'], true)) {
                            $p->status = $pi->status === 'requires_capture' ? 'authorized' : 'succeeded';

                            if ($p->status === 'succeeded' && Schema::hasColumn($p->getTable(), 'paid_at') && empty($p->paid_at)) {
                                $p->paid_at = now();
                            }

                            if (Schema::hasColumn($p->getTable(), 'provider_charge_id') && !empty($pi->latest_charge)) {
                                $p->provider_charge_id = $pi->latest_charge;
                            }

                            $p->save();
                        } else {
                            // Mirror live status even if not final yet
                            if ($this->hasColumns('payments', ['status'])) {
                                $p->status = $pi->status ?? $p->status;
                                $p->save();
                            }
                        }
                    } catch (\Throwable $e) {
                        Log::warning('Pay done() post-check failed: ' . $e->getMessage());
                    }
                }
            }
        }

        return view('pay.thanks', [
            'alreadyPaid' => session('status') === 'already_paid',
            'payment'     => $p ?? null,
            'token'       => $token ?? null,
        ]);
    }

    // ----------------- Helpers -----------------

    /** Check if all given columns exist on a table */
    protected function hasColumns(string $table, array $cols): bool
    {
        foreach ($cols as $c) {
            if (!Schema::hasColumn($table, $c)) {
                return false;
            }
        }
        return true;
    }

    /** Return the first non-empty attribute value from the model where the column exists */
    protected function firstExistingColumnValue($model, array $candidates): ?string
    {
        foreach ($candidates as $col) {
            if (Schema::hasColumn($model->getTable(), $col)) {
                $val = (string) ($model->{$col} ?? '');
                if ($val !== '') {
                    return $val;
                }
            }
        }
        return null;
    }

    /**
     * Resolve (or lazily create) a Stripe Customer for this payment.
     * Tries common columns on related Customer (e.g., stripe_customer_id / provider_customer_id),
     * falls back to email-based search.
     */
    protected function resolveStripeCustomerId(StripeClient $stripe, $payment): ?string
    {
        try {
            $customer = $payment->job->customer ?? null;

            // Prefer stored IDs on the Customer if your schema has them
            if ($customer) {
                $candidateCols = [
                    'stripe_customer_id',
                    'provider_customer_id',
                    'gateway_customer_id',
                ];
                foreach ($candidateCols as $col) {
                    if (Schema::hasColumn($customer->getTable(), $col)) {
                        $val = (string) ($customer->{$col} ?? '');
                        if ($val !== '') {
                            return $val;
                        }
                    }
                }
            }

            // If we have an email, try to find an existing Stripe customer by email
            $email = $customer->email ?? $payment->email ?? $payment->customer_email ?? null;
            if ($email) {
                $list = $stripe->customers->search([
                    'query' => "email:'" . addslashes($email) . "'",
                    'limit' => 1,
                ]);

                if (!empty($list->data) && isset($list->data[0]->id)) {
                    return (string) $list->data[0]->id;
                }
            }
        } catch (\Throwable $e) {
            Log::info('resolveStripeCustomerId failed: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * Check if the Stripe customer already has an off-session-capable payment method.
     */
    protected function hasOffSessionAuth(StripeClient $stripe, ?string $customerId): bool
    {
        if (!$customerId) return false;

        try {
            $cust = $stripe->customers->retrieve($customerId);
            if (!empty($cust->invoice_settings?->default_payment_method)) return true;

            $pms = $stripe->paymentMethods->all(['customer' => $customerId, 'type' => 'card', 'limit' => 1]);
            return count($pms->data ?? []) > 0;
        } catch (\Throwable $e) {
            Log::info('hasOffSessionAuth check failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Best-effort cancel of a PaymentIntent (used when we must recreate with manual capture).
     */
    protected function safeCancel(StripeClient $stripe, string $piId): void
    {
        try {
            $stripe->paymentIntents->cancel($piId);
        } catch (\Throwable $e) {
            Log::warning('safeCancel PI failed: ' . $e->getMessage(), ['pi' => $piId]);
        }
    }
}
