<?php

declare(strict_types=1);

namespace App\Http\Controllers\Portal;

use App\Http\Controllers\Controller;
use App\Mail\PaymentReceiptMail;
use App\Models\Deposit;
use App\Models\Job;
use App\Models\Payment;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Schema;
use Stripe\StripeClient;
use Throwable;

class PayController extends Controller
{
    /* ======================================================================
     | Stripe client
     * ====================================================================== */
    protected function stripe(): StripeClient
    {
        return new StripeClient((string) config('services.stripe.secret'));
    }

    /* ======================================================================
     | Helpers
     * ====================================================================== */

    protected function currencyFor(Job $job): string
    {
        $cur = $job->currency ?? optional($job->flow)->currency ?? 'NZD';
        return strtolower((string) $cur);
    }

    protected function validEmailOrNull(?string $email): ?string
    {
        $email = trim((string) $email);
        return $email !== '' && filter_var($email, FILTER_VALIDATE_EMAIL) ? $email : null;
    }

    protected function validCustomerEmail(Job $job): ?string
    {
        return $this->validEmailOrNull($job->customer_email)
            ?? $this->validEmailOrNull(optional($job->customer)->email);
    }

    /** Server-authoritative total to charge now (cents). */
    protected function chargeCentsResolved(Job $job): int
    {
        foreach ([
            (int) ($job->charge_amount_cents ?? 0),
            (int) ($job->amount_due_cents ?? 0),
            (int) ($job->charge_cents ?? 0),
            (int) ($job->total_cents ?? 0),
        ] as $v) {
            if ($v > 0) return $v;
        }
        if (!is_null($job->charge_amount) && $job->charge_amount > 0) {
            return (int) round(((float) $job->charge_amount) * 100);
        }
        if (!is_null($job->total_amount) && $job->total_amount > 0) {
            return (int) round(((float) $job->total_amount) * 100);
        }
        return 0;
    }

    /** Hold/bond amount (cents) with Job → Flow fallback. */
    protected function holdCentsResolved(Job $job): int
    {
        $job->loadMissing('flow');

        $jobVals  = [(int) ($job->hold_cents ?? 0), (int) ($job->hold_amount_cents ?? 0), (int) ($job->bond_cents ?? 0)];
        $flowVals = [
            (int) (optional($job->flow)->hold_cents ?? 0),
            (int) (optional($job->flow)->hold_amount_cents ?? 0),
            (int) (optional($job->flow)->bond_cents ?? 0),
            (int) (optional($job->flow)->deposit_cents ?? 0),
        ];
        foreach ($jobVals as $v)  { if ($v > 0) return $v; }
        foreach ($flowVals as $v) { if ($v > 0) return $v; }

        $maybe = $job->hold_amount ?? optional($job->flow)->hold_amount ?? null;
        if (!is_null($maybe) && (float) $maybe > 0) {
            return (int) round(((float) $maybe) * 100);
        }
        return 0;
    }

    /** Remaining after successful/processing non-hold payments (cents). */
    protected function remainingCents(Job $job): int
    {
        $total = $this->chargeCentsResolved($job);

        $paid = (int) ($job->amount_paid_cents
            ?? $job->paid_amount_cents
            ?? $job->paid_cents
            ?? 0);

        // Fallback: sum payments if job fields aren’t populated
        if ($paid === 0 && method_exists($job, 'payments')) {
            try {
                $col = Schema::hasColumn('payments', 'amount_cents') ? 'amount_cents'
                      : (Schema::hasColumn('payments', 'amount') ? 'amount' : null);
                if ($col) {
                    $paid = (int) $job->payments()
                        ->whereIn('status', ['succeeded', 'captured', 'paid', 'completed', 'processing'])
                        ->where(function ($q) {
                            $q->whereNull('purpose')->orWhere('purpose', '!=', 'hold');
                        })
                        ->sum($col);
                }
            } catch (Throwable $e) {
                Log::warning('remainingCents sum failed', ['job' => $job->id, 'err' => $e->getMessage()]);
            }
        }

        return max(0, $total - $paid);
    }

    /** Safe assign only if the column exists. */
    protected function safeSet(Model $model, string $column, $value): void
    {
        try {
            $table = method_exists($model, 'getTable') ? $model->getTable() : null;
            if ($table && Schema::hasColumn($table, $column)) {
                $model->{$column} = $value;
            }
        } catch (Throwable $e) {
            Log::warning('safeSet failed', [
                'table'  => method_exists($model, 'getTable') ? $model->getTable() : 'unknown',
                'column' => $column,
                'err'    => $e->getMessage(),
            ]);
        }
    }

    /** Recompute paid/balance & status after changes. */
    protected function recomputeJobMoney(Job $job): void
    {
        try {
            $job->refresh();

            $sum = 0;
            if (method_exists($job, 'payments')) {
                $col = Schema::hasColumn('payments', 'amount_cents') ? 'amount_cents'
                      : (Schema::hasColumn('payments', 'amount') ? 'amount' : null);
                if ($col) {
                    $sum = (int) $job->payments()
                        ->whereIn('status', ['succeeded', 'captured', 'paid', 'completed', 'processing'])
                        ->where(function ($q) {
                            $q->whereNull('purpose')->orWhere('purpose', '!=', 'hold');
                        })
                        ->sum($col);
                }
            }

            foreach (['amount_paid_cents','paid_amount_cents','paid_cents'] as $c) {
                if (Schema::hasColumn('jobs', $c)) { $job->{$c} = $sum; break; }
            }

            $remaining = max(0, $this->chargeCentsResolved($job) - $sum);
            if (Schema::hasColumn('jobs', 'balance_cents')) {
                $job->balance_cents = $remaining;
            }

            if (Schema::hasColumn('jobs', 'status')) {
                if ($remaining === 0 && $this->chargeCentsResolved($job) > 0) {
                    $job->status = 'active';
                } elseif ($sum > 0 && $remaining > 0) {
                    $job->status = 'partially_paid';
                }
            }
            $job->save();
        } catch (Throwable $e) {
            Log::warning('recomputeJobMoney failed', ['job' => $job->id, 'err' => $e->getMessage()]);
        }
    }

    /* ======================================================================
     | Views
     * ====================================================================== */

    public function showJob(Job $job, Request $request)
    {
        $bondHoldMode = $request->boolean('bond_hold', false);

        return view('portal.pay', [
            'job'            => $job,
            'clientSecret'   => null,
            'setupSecret'    => null,
            'offSessionAuth' => null,
            'currency'       => strtoupper($this->currencyFor($job)),
            'remainingCents' => $bondHoldMode ? $this->holdCentsResolved($job) : $this->remainingCents($job),
            'holdCents'      => $this->holdCentsResolved($job),
            'bondHoldMode'   => $bondHoldMode,
            'pk'             => config('services.stripe.key'),
        ]);
    }

    public function details(Request $request, Job $job)
    {
        return redirect()->route('portal.pay.show.job', ['job' => $job->id]);
    }

    public function show(Request $request, Job $job)
    {
        $job->loadMissing('flow');
        $bondHoldMode = $request->boolean('bond_hold', false);

        return view('portal.pay', [
            'job'            => $job,
            'currency'       => strtoupper($this->currencyFor($job)),
            'remainingCents' => $bondHoldMode ? $this->holdCentsResolved($job) : $this->remainingCents($job),
            'holdCents'      => $this->holdCentsResolved($job),
            'bondHoldMode'   => $bondHoldMode,
            'pk'             => config('services.stripe.key'),
        ]);
    }

    public function complete(Job $job, Request $request)
    {
        $clientSecret = $request->string('payment_intent_client_secret')->toString() ?: null;
        $setupSecret  = $request->string('setup_intent_client_secret')->toString() ?: null;

        $payment = Payment::where('job_id', $job->id)->latest('id')->first();
        $deposit = Deposit::where('job_id', $job->id)->latest('id')->first();

        return view('portal.pay-complete-job', compact('job','clientSecret','setupSecret','payment','deposit'));
    }

    /* ======================================================================
     | Intents (Payment Element)
     * ====================================================================== */

    /**
     * Create/update a PaymentIntent for the remaining balance,
     * or prepare a manual-capture bond hold PI (bond_hold=true).
     * For bond holds we DO NOT confirm server-side; the client confirms it.
     */
    public function intent(Request $request, Job $job): JsonResponse
    {
        try {
            $stripe       = $this->stripe();
            $bondHoldMode = $request->boolean('bond_hold', false);
            $onpageOnly   = $request->boolean('onpage_only', false);

            // --- Bond Hold Mode ---
            if ($bondHoldMode) {
                $job->loadMissing('flow');

                $amount = $this->holdCentsResolved($job);
                if ($amount < 50) {
                    return response()->json(['ok' => false, 'message' => 'No bond/hold configured for this booking.'], 422);
                }

                // Refuse if already have an authorised/captured hold
                $existingHold = Deposit::query()
                    ->where('job_id', $job->id)
                    ->whereIn('status', ['authorized','authorised','captured'])
                    ->first();

                if (! $existingHold && method_exists($job, 'payments')) {
                    $existingHold = $job->payments()
                        ->when(Schema::hasColumn('payments','purpose'), fn($q) => $q->where('purpose','hold'))
                        ->whereIn('status', ['authorized', 'captured'])
                        ->first();
                }

                if ($existingHold) {
                    return response()->json([
                        'ok'      => false,
                        'message' => 'A bond/hold is already on this booking.',
                        'reason'  => 'existing_hold',
                    ], 409);
                }

                // Ensure/reuse customer
                $receipt    = $this->validEmailOrNull($request->input('receipt_email')) ?? $this->validCustomerEmail($job);
                $customerId = $job->psp_customer_id ?: null;

                if (!$customerId) {
                    $sc = $stripe->customers->create([
                        'name'     => $job->customer_name ?: null,
                        'email'    => $receipt,
                        'metadata' => ['job_id' => (string) $job->id],
                    ]);
                    if (Schema::hasColumn('jobs', 'psp_customer_id')) {
                        $job->psp_customer_id = $sc->id;
                        $job->save();
                    }
                    $customerId = $sc->id;
                }

                // Create manual-capture hold PI (client will confirm)
                $pi = $stripe->paymentIntents->create(
                    [
                        'amount'     => $amount,
                        'currency'   => $this->currencyFor($job),
                        'customer'   => $customerId,
                        'capture_method' => 'manual',
                        'setup_future_usage' => 'off_session',
                        'automatic_payment_methods' => [
                            'enabled' => true,
                            'allow_redirects' => $onpageOnly ? 'never' : 'always',
                        ],
                        'metadata'   => [
                            'job_id'  => (string) $job->id,
                            'purpose' => 'bond_hold',
                            'env'     => app()->environment(),
                            'source'  => 'portal',
                        ],
                        'description'   => "Job #{$job->id} refundable bond/hold",
                        'receipt_email' => $receipt ?: null,
                    ],
                    ['idempotency_key' => "intent_bond_hold_job_{$job->id}_" . uniqid('', true)]
                );

                return response()->json([
                    'ok'            => true,
                    'client_secret' => $pi->client_secret,
                    'id'            => $pi->id,
                    'bond_hold'     => true,
                ]);
            }

            // --- Normal Payment Mode ---
            $amount = $this->remainingCents($job);
            if ($amount < 50) {
                return response()->json(['ok' => false, 'message' => 'Nothing to charge.'], 422);
            }

            $job->loadMissing('flow');
            $receipt    = $this->validEmailOrNull($request->input('receipt_email')) ?? $this->validCustomerEmail($job);
            $customerId = $job->psp_customer_id ?: null;

            if (!$customerId) {
                $sc = $stripe->customers->create([
                    'name'     => $job->customer_name ?: null,
                    'email'    => $receipt,
                    'metadata' => ['job_id' => (string) $job->id],
                ]);
                if (Schema::hasColumn('jobs', 'psp_customer_id')) {
                    $job->psp_customer_id = $sc->id;
                    $job->save();
                }
                $customerId = $sc->id;
            }

            $existingPi = trim((string) $request->input('payment_intent', ''));
            $params = [
                'amount'   => $amount,
                'currency' => $this->currencyFor($job),
                'customer' => $customerId,
                'setup_future_usage' => 'off_session',
                'automatic_payment_methods' => [
                    'enabled' => true,
                    'allow_redirects' => $onpageOnly ? 'never' : 'always',
                ],
                'metadata'    => [
                    'job_id' => (string) $job->id,
                    'env'    => app()->environment(),
                    'source' => 'portal',
                ],
                'description' => "Job #{$job->id} payment",
            ];
            if ($receipt) {
                $params['receipt_email'] = $receipt;
            }

            $pi = $existingPi !== ''
                ? $stripe->paymentIntents->update(
                    $existingPi,
                    Arr::only($params, [
                        'amount','metadata','description','receipt_email',
                        'automatic_payment_methods','setup_future_usage','customer'
                    ])
                )
                : $stripe->paymentIntents->create(
                    $params,
                    ['idempotency_key' => "intent_job_{$job->id}_" . uniqid('', true)]
                );

            return response()->json([
                'ok'            => true,
                'client_secret' => $pi->client_secret,
                'id'            => $pi->id,
            ]);
        } catch (Throwable $e) {
            Log::error('PayController@intent failed', [
                'job' => $job->id ?? null,
                'err' => $e->getMessage(),
            ]);
            return response()->json(['ok' => false, 'message' => 'Unable to create payment intent'], 422);
        }
    }

    /* ======================================================================
     | Bundle (final charge only; hold can be created on-page or after charge)
     * ====================================================================== */

    /**
     * Create the PaymentIntent for the final payment, or a hold-only PI when no charge.
     * - If charge present: returns { payment: { id, client_secret } }
     * - If charge absent & hold present: returns { hold: { id, client_secret } }
     */
    public function bundle(Request $request, Job $job): JsonResponse
    {
        $job->loadMissing('flow');

        try {
            $stripe   = $this->stripe();
            $currency = $this->currencyFor($job);
            $receipt  = $this->validCustomerEmail($job);
            $onpage   = $request->boolean('onpage_only', false);

            $chargeCts = (int) $request->input('requested_charge_cents', 0);
            $holdCts   = (int) $request->input('requested_hold_cents', 0);

            if ($chargeCts <= 0) $chargeCts = $this->remainingCents($job);
            if ($holdCts   <= 0) $holdCts   = $this->holdCentsResolved($job);

            if ($chargeCts < 50 && $holdCts < 50) {
                return response()->json(['ok' => false, 'message' => 'Nothing to charge.'], 422);
            }

            // Ensure/reuse Stripe Customer
            $customerId = $job->psp_customer_id ?: null;
            if (!$customerId) {
                $sc = $stripe->customers->create([
                    'name'     => $job->customer_name ?: null,
                    'email'    => $receipt,
                    'metadata' => ['job_id' => (string) $job->id],
                ]);
                if (Schema::hasColumn('jobs','psp_customer_id')) {
                    $job->psp_customer_id = $sc->id;
                    $job->save();
                }
                $customerId = $sc->id;
            }

            // Case A: charge now
            if ($chargeCts >= 50) {
                $pi = $stripe->paymentIntents->create(
                    [
                        'amount'     => $chargeCts,
                        'currency'   => $currency,
                        'customer'   => $customerId,
                        'setup_future_usage' => 'off_session',
                        'automatic_payment_methods' => [
                            'enabled' => true,
                            'allow_redirects' => $onpage ? 'never' : 'always',
                        ],
                        'metadata'    => [
                            'job_id'  => (string) $job->id,
                            'purpose' => 'charge',
                            'env'     => app()->environment(),
                            'source'  => 'portal',
                        ],
                        'description'   => "Job #{$job->id} final payment",
                        'receipt_email' => $receipt ?: null,
                    ],
                    ['idempotency_key' => "bundle_payment_only_{$job->id}_" . uniqid('', true)]
                );

                return response()->json([
                    'ok'      => true,
                    'payment' => ['id' => $pi->id, 'client_secret' => $pi->client_secret],
                ]);
            }

            // Case B: hold-only
            if ($holdCts >= 50) {
                $holdPi = $stripe->paymentIntents->create(
                    [
                        'amount'     => $holdCts,
                        'currency'   => $currency,
                        'customer'   => $customerId,
                        'capture_method' => 'manual',
                        'setup_future_usage' => 'off_session',
                        'automatic_payment_methods' => [
                            'enabled' => true,
                            'allow_redirects' => $onpage ? 'never' : 'always',
                        ],
                        'metadata'    => [
                            'job_id'  => (string) $job->id,
                            'purpose' => 'bond_hold',
                            'env'     => app()->environment(),
                            'source'  => 'portal',
                        ],
                        'description'   => "Job #{$job->id} refundable bond/hold",
                        'receipt_email' => $receipt ?: null,
                    ],
                    ['idempotency_key' => "bundle_hold_only_{$job->id}_" . uniqid('', true)]
                );

                return response()->json([
                    'ok'   => true,
                    'hold' => ['id' => $holdPi->id, 'client_secret' => $holdPi->client_secret],
                ]);
            }

            return response()->json(['ok' => false, 'message' => 'Nothing to charge.'], 422);

        } catch (\Stripe\Exception\ApiErrorException $e) {
            Log::error('PayController@bundle Stripe', ['job' => $job->id, 'msg' => $e->getMessage()]);
            return response()->json(['ok' => false, 'message' => $e->getMessage()], 422);
        } catch (\Throwable $e) {
            Log::error('PayController@bundle error', ['job' => $job->id, 'msg' => $e->getMessage()]);
            return response()->json(['ok' => false, 'message' => 'Server error preparing payment'], 500);
        }
    }

    /* ======================================================================
     | Persist results
     * ====================================================================== */

    /**
     * Called by the client after confirming either:
     *  - a final payment (payment_intent), or
     *  - a hold-only intent (hold_intent).
     * Persists the Payment and Deposit rows; auto-creates a hold after charge
     * only if the client did not already send a hold_intent.
     */
    public function recordPaid(Job $job, Request $request)
    {
        $data = $request->validate([
            'payment_intent'       => ['nullable','string','max:191'],
            'hold_intent'          => ['nullable','string','max:191'],
            'requested_hold_cents' => ['nullable','integer','min:0'],
            'status'               => ['nullable','string','max:50'],
            'receipt_email'        => ['nullable','string','max:191'],
        ]);

        // --- SAFETY NET: normalize misrouted hold as payment_intent ---
        $holdIntentId    = $data['hold_intent'] ?? null;
        $paymentIntentId = $data['payment_intent'] ?? null;

        if (!$holdIntentId && $paymentIntentId) {
            try {
                $piProbe  = $this->stripe()->paymentIntents->retrieve($paymentIntentId, []);
                $purpose  = (string) ($piProbe->metadata['purpose'] ?? '');
                $isManual = ($piProbe->capture_method === 'manual')
                         && ($purpose === 'bond_hold' || $purpose === 'hold' || ((int)($piProbe->amount_capturable ?? 0) > 0));
                if ($isManual) {
                    $holdIntentId = $paymentIntentId;
                    unset($data['payment_intent']);
                    $data['hold_intent'] = $holdIntentId;
                }
            } catch (\Throwable $e) {
                // non-fatal
            }
        }

        $stripe = $this->stripe();

        try {
            DB::transaction(function () use ($stripe, $job, $data) {
                $pi = null;
                $payment = null;

                // ---- 1) Persist the final charge (Payment)
                if (!empty($data['payment_intent'])) {
                    $pi = $stripe->paymentIntents->retrieve(
                        $data['payment_intent'],
                        ['expand' => ['latest_charge']]
                    );

                    $persistStatus = match ($pi->status) {
                        'succeeded'        => 'succeeded',
                        'processing'       => 'processing',
                        'requires_capture' => 'processing',
                        default            => $pi->status,
                    };

                    $payment = Payment::firstOrNew(['stripe_payment_intent' => $pi->id]);

                    if (Schema::hasColumn('payments','amount_cents')) {
                        $payment->amount_cents = (int) $pi->amount;
                    } elseif (Schema::hasColumn('payments','amount')) {
                        $payment->amount = (int) $pi->amount;
                    }

                    $payment->job_id        = $job->id;
                    if (Schema::hasColumn('payments','currency')) {
                        $payment->currency = strtoupper($pi->currency ?? ($job->currency ?? 'NZD'));
                    }
                    if (Schema::hasColumn('payments','type')) {
                        $payment->type = 'charge';
                    }
                    if (Schema::hasColumn('payments','provider')) {
                        $payment->provider = 'stripe';
                    }
                    $payment->status        = $persistStatus;
                    $payment->stripe_charge = $pi->latest_charge?->id ?? null;
                    if (Schema::hasColumn('payments','reference')) {
                        $payment->reference = $pi->latest_charge?->id ?? null;
                    }
                    if (Schema::hasColumn('payments','meta')) {
                        $payment->meta = ['from' => 'portal.recordPaid'];
                    }
                    $payment->save();

                    if ($pi->customer && Schema::hasColumn('jobs','psp_customer_id') && empty($job->psp_customer_id)) {
                        $job->psp_customer_id = $pi->customer;
                        $job->save();
                    }

                    // Send receipt
                    $postedEmail = isset($data['receipt_email']) ? (string) $data['receipt_email'] : null;
                    $this->sendPaymentReceipt($job, $payment, $pi, $postedEmail);
                }

                // ---- 2) Create/persist the hold if sent by client
                $desiredHold = isset($data['requested_hold_cents'])
                    ? max(0, (int) $data['requested_hold_cents'])
                    : $this->holdCentsResolved($job);

                if (!empty($data['hold_intent'])) {
                    try {
                        $holdPi = $stripe->paymentIntents->retrieve($data['hold_intent'], []);
                        $this->upsertDepositFor($job, $holdPi);
                        $this->upsertBondHoldPaymentFor($job, $holdPi);
                    } catch (Throwable $e) {
                        Log::warning('recordPaid: hold_intent retrieve failed', ['job' => $job->id, 'err' => $e->getMessage()]);
                    }
                }

                // ---- 3) Optional auto-create hold AFTER a charge when needed
                if ($desiredHold >= 50) {
                    $existing = Deposit::query()
                        ->where('job_id', $job->id)
                        ->whereIn('status', ['authorized', 'authorised', 'captured'])
                        ->first();

                    // Only auto-create when:
                    // - no existing hold,
                    // - we processed a real charge ($pi),
                    // - and client did NOT already send a hold_intent
                    if (!$existing && $pi && empty($data['hold_intent'])) {
                        $pm         = $pi->payment_method ?? ($pi->latest_charge?->payment_method ?? null);
                        $customerId = $job->psp_customer_id ?: ($pi->customer ?: null);

                        if ($pm && $customerId) {
                            $hold = $stripe->paymentIntents->create([
                                'amount'               => $desiredHold,
                                'currency'             => $this->currencyFor($job),
                                'customer'             => $customerId,
                                'payment_method'       => $pm,
                                'payment_method_types' => ['card'],
                                'capture_method'       => 'manual',
                                'confirm'              => true,
                                'off_session'          => true,
                                'metadata'             => [
                                    'job_id'  => (string) $job->id,
                                    'purpose' => 'hold',
                                    'env'     => app()->environment(),
                                    'source'  => 'portal',
                                ],
                                'description'          => "Job #{$job->id} refundable security hold",
                            ], [
                                'idempotency_key' => "auto_hold_after_charge_job_{$job->id}_{$pi->id}",
                            ]);

                            $this->upsertDepositFor($job, $hold);
                            $this->upsertBondHoldPaymentFor($job, $hold);
                        }
                    }
                }

                // ---- 4) Recompute money / status
                $this->recomputeJobMoney($job);

                Log::info('recordPaid persisted', [
                    'job'        => $job->id,
                    'payment_pi' => $data['payment_intent'] ?? null,
                    'hold_pi'    => $data['hold_intent'] ?? null,
                ]);
            });

            return response()->json(['ok' => true]);
        } catch (\Throwable $e) {
            Log::error('recordPaid failed', ['job' => $job->id, 'err' => $e->getMessage()]);
            return response()->json(['error' => 'record-failed', 'message' => $e->getMessage()], 500);
        }
        // ⛔️ Do NOT put any code here — execution already returned above.
    }

    /* ======================================================================
     | Test email helper
     * ====================================================================== */

    public function _testSend($jobId, $paymentId, $to)
    {
        $job = Job::findOrFail($jobId);
        $payment = Payment::findOrFail($paymentId);
        $this->sendPaymentReceipt($job, $payment, null, (string) $to);
        return 'ok';
    }

    /* ======================================================================
     | Deposit upsert
     * ====================================================================== */

    /** Create/update a Deposit row from a Stripe PaymentIntent (manual capture or captured). */
    protected function upsertDepositFor(Job $job, \Stripe\PaymentIntent $spi)
    {
        if (!Schema::hasTable('deposits')) { return null; }

        $authCol = Schema::hasColumn('deposits','authorized_cents') ? 'authorized_cents'
                 : (Schema::hasColumn('deposits','authorised_cents') ? 'authorised_cents'
                 : (Schema::hasColumn('deposits','amount_cents') ? 'amount_cents' : null));
        $captCol = Schema::hasColumn('deposits','captured_cents') ? 'captured_cents'
                 : (Schema::hasColumn('deposits','captured_amount_cents') ? 'captured_amount_cents' : null);
        $refCol  = Schema::hasColumn('deposits','reference') ? 'reference'
                 : (Schema::hasColumn('deposits','reference_number') ? 'reference_number'
                 : (Schema::hasColumn('deposits','stripe_payment_intent') ? 'stripe_payment_intent' : null));

        $q = Deposit::query();
        if ($refCol) { $q->where($refCol, $spi->id); }
        $deposit = $q->first() ?: new Deposit();

        if (Schema::hasColumn('deposits','job_id'))      { $deposit->job_id = $job->id; }
        if (Schema::hasColumn('deposits','booking_id'))  { $deposit->booking_id = $job->booking_id ?? null; }
        if (Schema::hasColumn('deposits','customer_id')) { $deposit->customer_id = $job->customer_id ?? null; }
        if (Schema::hasColumn('deposits','brand_id'))    { $deposit->brand_id = $job->brand_id ?? null; }

        $amount      = (int) ($spi->amount ?? 0);
        $currency    = strtoupper($spi->currency ?? ($job->currency ?? 'NZD'));
        $wasCaptured = ((int)($spi->amount_capturable ?? 0) === 0 && $spi->status === 'succeeded');

        if ($authCol) { $deposit->{$authCol} = $amount; }
        if ($captCol) { $deposit->{$captCol} = $wasCaptured ? $amount : 0; }

        if (Schema::hasColumn('deposits','currency')) { $deposit->currency = $currency; }
        if ($refCol) { $deposit->{$refCol} = $spi->id; }
        if (Schema::hasColumn('deposits','status')) { $deposit->status = $wasCaptured ? 'captured' : 'authorized'; }
        if (Schema::hasColumn('deposits','type'))   { $deposit->type = 'deposit'; }
        if (Schema::hasColumn('deposits','stripe_payment_intent')) { $deposit->stripe_payment_intent = $spi->id; }
        if (Schema::hasColumn('deposits','stripe_payment_method')) { $deposit->stripe_payment_method = $spi->payment_method ?? null; }
        if (Schema::hasColumn('deposits','meta'))   { $deposit->meta = ['from' => 'portal.recordPaidVerified']; }

        try {
            $deposit->save();
        } catch (\Illuminate\Database\QueryException $e) {
            foreach (['authorized_cents','authorised_cents','amount_cents','captured_cents','captured_amount_cents'] as $col) {
                if (!Schema::hasColumn('deposits', $col) && isset($deposit->{$col})) { unset($deposit->{$col}); }
            }
            $deposit->save();
        }

        return $deposit;
    }

    /* ======================================================================
     | Payment (bond_hold) upsert — visible in Payments list
     * ====================================================================== */

    /**
     * Create/update a Payment row for a manual-capture hold PI.
     * We intentionally use type='charge' + purpose='hold' so it appears
     * in the Payments admin list (which filters on known types), while
     * your money sums continue to exclude it via purpose='hold'.
     */
    protected function upsertBondHoldPaymentFor(Job $job, \Stripe\PaymentIntent $spi): ?Payment
    {
        try {
            $query = Payment::query();
            if (Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
                $query->where('stripe_payment_intent_id', $spi->id);
            } elseif (Schema::hasColumn('payments', 'stripe_payment_intent')) {
                $query->where('stripe_payment_intent', $spi->id);
            }
            $payment = $query->first() ?: new Payment();

            // Set basic identifiers
            $payment->job_id = $job->id;

            if (Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
                $payment->stripe_payment_intent_id = $spi->id;
            }
            if (Schema::hasColumn('payments', 'stripe_payment_intent')) {
                $payment->stripe_payment_intent = $spi->id;
            }
            if (Schema::hasColumn('payments', 'stripe_payment_method_id') && !empty($spi->payment_method)) {
                $payment->stripe_payment_method_id = is_string($spi->payment_method) ? $spi->payment_method : null;
            }
            if (Schema::hasColumn('payments', 'stripe_payment_method') && !empty($spi->payment_method)) {
                $payment->stripe_payment_method = is_string($spi->payment_method) ? $spi->payment_method : null;
            }

            // Amounts
            $amount = (int) ($spi->amount ?? 0);
            if (Schema::hasColumn('payments', 'amount_cents')) {
                $payment->amount_cents = $amount;
            } elseif (Schema::hasColumn('payments', 'amount')) {
                $payment->amount = $amount;
            }

            // Visibility + classification
            if (Schema::hasColumn('payments', 'currency')) {
                $payment->currency = strtoupper($spi->currency ?? ($job->currency ?? 'NZD'));
            }
            if (Schema::hasColumn('payments', 'type')) {
                $payment->type = 'charge';        // make it visible in Payments list
            }
            if (Schema::hasColumn('payments', 'purpose')) {
                $payment->purpose = 'hold';        // keep out of paid totals
            }
            if (Schema::hasColumn('payments', 'provider')) {
                $payment->provider = 'stripe';
            }
            if (Schema::hasColumn('payments', 'psp')) {
                $payment->psp = 'stripe';
            }
            if (Schema::hasColumn('payments', 'gateway')) {
                $payment->gateway = 'stripe';
            }
            if (Schema::hasColumn('payments', 'mechanism')) {
                $payment->mechanism = 'card';
            }
            if (Schema::hasColumn('payments', 'capture_method')) {
                $payment->capture_method = $spi->capture_method ?? 'manual';
            }
            if (Schema::hasColumn('payments', 'authorized_at') && empty($payment->authorized_at)) {
                $payment->authorized_at = now();
            }

            // Status mapping for holds
            $status = (string) ($spi->status ?? '');
            if (Schema::hasColumn('payments', 'status')) {
                if ($status === 'requires_capture') {
                    $payment->status = 'authorized';
                } elseif ($status === 'succeeded' && (int)($spi->amount_capturable ?? 0) === 0) {
                    $payment->status = 'captured';
                    if (Schema::hasColumn('payments', 'captured_at') && empty($payment->captured_at)) {
                        $payment->captured_at = now();
                    }
                } elseif ($status === 'canceled') {
                    $payment->status = 'canceled';
                } else {
                    $payment->status = $status ?: 'authorized';
                }
            }

            if (Schema::hasColumn('payments', 'reference') && empty($payment->reference)) {
                $payment->reference = $spi->id;
            }
            if (Schema::hasColumn('payments', 'meta')) {
                $payment->meta = array_merge((array) ($payment->meta ?? []), [
                    'from'    => 'portal.recordPaidVerified',
                    'purpose' => 'hold',
                ]);
            }
            if (Schema::hasColumn('payments', 'gateway_meta')) {
                $payment->gateway_meta = ['pi' => $spi->id];
            }

            $payment->save();

            return $payment;
        } catch (Throwable $e) {
            Log::error('upsertBondHoldPaymentFor failed', ['job' => $job->id, 'err' => $e->getMessage()]);
            return null;
        }
    }

    /* ======================================================================
     | Receipt email sender
     * ====================================================================== */

    /**
     * Send a payment receipt using robust fallbacks for the recipient address.
     * Order tried:
     *  - job->customer_email
     *  - posted $postedEmail
     *  - PI->receipt_email
     *  - PI->latest_charge->billing_details->email
     * Also persists the chosen email back to the Job if it's empty.
     */
    protected function sendPaymentReceipt(Job $job, Payment $payment, $piOrId = null, ?string $postedEmail = null): void
    {
        try {
            $stripe = $this->stripe();
            $spi = null;

            if (is_string($piOrId) && $piOrId !== '') {
                try { $spi = $stripe->paymentIntents->retrieve($piOrId, ['expand' => ['latest_charge']]); } catch (\Throwable $e) {}
            } elseif ($piOrId instanceof \Stripe\PaymentIntent) {
                $spi = $piOrId;
            }

            $candidates = [
                $this->validEmailOrNull($job->customer_email),
                $this->validEmailOrNull($postedEmail),
                $this->validEmailOrNull($spi->receipt_email ?? null),
                $this->validEmailOrNull($spi->latest_charge->billing_details->email ?? null),
            ];

            $to = null;
            foreach ($candidates as $em) {
                if ($em) { $to = $em; break; }
            }
            if (! $to) {
                Log::warning('No valid receipt email found; skipping send', ['job' => $job->id, 'payment' => $payment->id]);
                return;
            }

            if (empty($job->customer_email)) {
                try { $job->customer_email = $to; $job->save(); } catch (\Throwable $e) {}
            }

            Mail::to($to)->send(new PaymentReceiptMail($job, $payment));
            Log::info('PaymentReceiptMail sent', ['job_id' => $job->id, 'payment_id' => $payment->id, 'to' => $to]);
        } catch (\Throwable $e) {
            Log::error('PaymentReceiptMail FAILED', ['job_id' => $job->id ?? null, 'payment_id' => $payment->id ?? null, 'error' => $e->getMessage()]);
        }
    }
}
