<?php
// app/Http/Controllers/StripeWebhookController.php

declare(strict_types=1);

namespace App\Http\Controllers;

use App\Events\PaymentSucceeded;
use App\Models\Payment;
use App\Models\Deposit;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Symfony\Component\HttpFoundation\Response;
use Stripe\Webhook as StripeWebhook;

class StripeWebhookController extends Controller
{
    /**
     * Stripe webhook endpoint.
     * Verifies signature and processes events.
     */
    public function handle(Request $request): Response
    {
        $secret = (string) config('services.stripe.webhook_secret');
        $sig    = $request->server('HTTP_STRIPE_SIGNATURE');

        // Verify signature
        try {
            $event = StripeWebhook::constructEvent(
                $request->getContent(),
                $sig,
                $secret
            );
        } catch (\UnexpectedValueException $e) {
            Log::warning('[Stripe webhook] Invalid payload', ['error' => $e->getMessage()]);
            return response('Invalid payload', 400);
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            Log::warning('[Stripe webhook] Invalid signature', ['error' => $e->getMessage()]);
            return response('Invalid signature', 400);
        }

        $type   = (string) ($event->type ?? 'unknown');
        $object = (array) ($event->data->object ?? []);
        Log::info('[Stripe webhook] ' . $type, ['id' => $event->id ?? null]);

        try {
            switch ($type) {
                /**
                 * ---- Deposit / Manual Hold Events ----
                 */
                case 'payment_intent.amount_capturable_updated': {
                    // Fires when a manual-capture PI is authorised (funds held).
                    $piId      = (string) Arr::get($object, 'id');
                    $depositId = Arr::get($object, 'metadata.deposit_id');

                    if ($depositId && ($d = Deposit::find($depositId))) {
                        $update = [
                            'status'                   => 'requires_capture',
                            'stripe_payment_intent_id' => $piId,
                        ];
                        if (Schema::hasColumn('deposits', 'deposit_confirmed_at') && empty($d->deposit_confirmed_at)) {
                            $update['deposit_confirmed_at'] = now();
                        }
                        if (Schema::hasColumn('deposits', 'stripe_payment_method')) {
                            $update['stripe_payment_method'] = Arr::get($object, 'payment_method');
                        }
                        $d->forceFill($update)->save();
                    }
                    break;
                }

                case 'payment_intent.succeeded': {
                    // For manual capture, succeeded means the hold was captured.
                    $piId      = (string) Arr::get($object, 'id');
                    $depositId = Arr::get($object, 'metadata.deposit_id');

                    if ($depositId && ($d = Deposit::find($depositId))) {
                        $update = ['status' => 'captured'];
                        if (Schema::hasColumn('deposits', 'captured_at')) {
                            $update['captured_at'] = now();
                        }
                        if (Schema::hasColumn('deposits', 'stripe_payment_intent_id')) {
                            $update['stripe_payment_intent_id'] = $piId;
                        }
                        $d->forceFill($update)->save();
                    }

                    // ---- Existing Payment flow ----
                    $paymentId = Arr::get($object, 'metadata.payment_id');
                    if ($paymentId && ($p = Payment::find($paymentId))) {
                        if ($p->status !== 'succeeded') {
                            $meta       = (array) ($p->gateway_meta ?? []);
                            $meta['pi'] = $object['id'] ?? ($meta['pi'] ?? null);

                            $p->forceFill([
                                'status'       => 'succeeded',
                                'paid_at'      => $p->paid_at ?? now(),
                                'gateway_meta' => $meta,
                            ])->save();

                            event(new PaymentSucceeded($p));
                        }
                    }
                    break;
                }

                /**
                 * ---- Mirror PI state to Payment (processing/failed/canceled/requires_action) ----
                 * This keeps payments.status aligned with Stripe's PI status and captures useful details.
                 */
                case 'payment_intent.processing':
                case 'payment_intent.payment_failed':
                case 'payment_intent.canceled':
                case 'payment_intent.requires_action': {
                    $piId      = (string) Arr::get($object, 'id');
                    $paymentId = Arr::get($object, 'metadata.payment_id');

                    if ($paymentId && ($p = Payment::find($paymentId))) {
                        $update = [
                            'status' => (string) ($object['status'] ?? $p->status),
                        ];

                        // Merge last_payment_error into a structured "details" column if present
                        $lastErr = Arr::get($object, 'last_payment_error');
                        if (!empty($lastErr)) {
                            $details = (array) ($p->details ?? []);
                            $details['last_payment_error'] = $lastErr;
                            if (Schema::hasColumn('payments', 'details')) {
                                $update['details'] = $details;
                            } else {
                                // Fallback: keep previous behavior of stashing a message in gateway_meta
                                $meta               = (array) ($p->gateway_meta ?? []);
                                $meta['last_error'] = Arr::get($lastErr, 'message');
                                $update['gateway_meta'] = $meta;
                            }
                        }

                        // Save latest_charge as provider_charge_id if that column exists
                        if (!empty($object['latest_charge']) && Schema::hasColumn('payments', 'provider_charge_id')) {
                            $update['provider_charge_id'] = (string) $object['latest_charge'];
                        }

                        // Ensure we remember the PI in gateway_meta (nice to have)
                        $meta       = (array) ($p->gateway_meta ?? []);
                        $meta['pi'] = $piId ?: ($meta['pi'] ?? null);
                        $update['gateway_meta'] = $meta;

                        $p->forceFill($update)->save();
                    }

                    // Keep deposit release behavior on canceled PIs (manual-capture holds).
                    if ($type === 'payment_intent.canceled') {
                        $depositId = Arr::get($object, 'metadata.deposit_id');
                        if ($depositId && ($d = Deposit::find($depositId))) {
                            $d->forceFill(['status' => 'released'])->save();
                        }
                    }

                    break;
                }

                /**
                 * ---- Checkout session flow ----
                 */
                case 'checkout.session.completed': {
                    $paymentId = Arr::get($object, 'metadata.payment_id');
                    if ($paymentId && ($p = Payment::find($paymentId))) {
                        if ($p->status !== 'succeeded') {
                            $meta                     = (array) ($p->gateway_meta ?? []);
                            $meta['checkout_session'] = $object['id'] ?? ($meta['checkout_session'] ?? null);
                            $meta['pi']               = $object['payment_intent'] ?? ($meta['pi'] ?? null);

                            $p->forceFill([
                                'status'       => 'succeeded',
                                'paid_at'      => $p->paid_at ?? now(),
                                'gateway_meta' => $meta,
                            ])->save();

                            event(new PaymentSucceeded($p));
                        }
                    }
                    break;
                }

                default:
                    Log::info('[Stripe webhook] Unhandled event', ['type' => $type]);
            }
        } catch (\Throwable $e) {
            Log::error('[Stripe webhook] Handler error', [
                'type'  => $type,
                'err'   => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response('ok', 200);
        }

        return response('ok', 200);
    }
}
