<?php
// app/Http/Controllers/VevsWebhookController.php

declare(strict_types=1);

namespace App\Http\Controllers;

use App\Jobs\UpsertBookingFromVeVs;
use App\Jobs\ImportVevsReservation;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;

class VevsWebhookController extends Controller
{
    /**
     * Minimal entrypoint (if you wire the route to __invoke).
     */
    public function __invoke(Request $request): JsonResponse
    {
        Log::info('[VEVS webhook] hit', ['raw' => $request->getContent()]);
        return $this->handle($request);
    }

    /**
     * Handle VEVS webhook deliveries.
     *
     * Behaviors:
     *  - If a reservation-style payload includes a ref_id (top-level or reservation.ref_id),
     *    dispatch ImportVevsReservation(ref_id).
     *  - If booking rows are present (in data / bookings / single object), upsert them via
     *    UpsertBookingFromVeVs (queued or inline depending on config).
     */
    public function handle(Request $request): JsonResponse
    {
        $requestId = (string) str()->uuid();

        Log::info('[VEVS webhook] hit', [
            'ip'  => $request->ip(),
            'rid' => $requestId,
        ]);

        // 1) Verify secret (optional)
        $configuredSecret = (string) (config('services.vevs.webhook_secret') ?? '');
        if ($configuredSecret !== '') {
            $provided = trim((string) ($request->header('X-Vevs-Secret') ?? $request->query('token') ?? ''));
            if (! hash_equals($configuredSecret, $provided)) {
                Log::warning('[VEVS webhook] Invalid secret', ['rid' => $requestId]);
                return response()->json([
                    'ok'    => false,
                    'rid'   => $requestId,
                    'error' => 'Invalid signature/secret',
                ], 401);
            }
        }

        // 2) Normalize payload
        $payload = $this->parsePayload($request);
        if ($payload === null) {
            return response()->json([
                'ok'    => false,
                'rid'   => $requestId,
                'error' => 'Empty or invalid payload',
            ], 400);
        }

        // Decide queue/inline once
        $useQueue = filter_var(config('services.vevs.queue'), FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
        $useQueue = $useQueue ?? true;

        // 3) If this is a reservation-created/updated style payload (contains a ref_id),
        //    dispatch the ImportVevsReservation job.
        $ref = Arr::get($payload, 'ref_id') ?? Arr::get($payload, 'reservation.ref_id');
        $importDispatched = false;

        if (is_string($ref) && trim($ref) !== '') {
            $ref = trim($ref);
            try {
                if ($useQueue) {
                    ImportVevsReservation::dispatch($ref);
                } else {
                    // Run inline for debugging
                    (new ImportVevsReservation($ref))->handle();
                }
                $importDispatched = true;
                Log::info('[VEVS webhook] ImportVevsReservation dispatched', ['rid' => $requestId, 'ref_id' => $ref]);
            } catch (\Throwable $e) {
                Log::error('[VEVS webhook] ImportVevsReservation failed', [
                    'rid'    => $requestId,
                    'ref_id' => $ref,
                    'error'  => $e->getMessage(),
                ]);
            }
        }

        // 4) Extract booking rows (if present) and upsert
        $rows = [];
        $candidate = Arr::get($payload, 'data') ?? Arr::get($payload, 'bookings');
        if (is_array($candidate)) {
            $rows = array_values(array_filter($candidate, 'is_array'));
        } elseif (is_array($payload) && array_keys($payload) !== range(0, count($payload) - 1)) {
            // single assoc object
            $rows = [$payload];
        }

        $processed = 0;
        $failed = 0;
        $errors = [];

        foreach ($rows as $idx => $row) {
            try {
                $row = (array) $row;
                if ($useQueue) {
                    UpsertBookingFromVeVs::dispatch($row);
                } else {
                    (new UpsertBookingFromVeVs($row))->handle();
                }
                $processed++;
            } catch (\Throwable $e) {
                $failed++;
                $errors[] = ['index' => $idx, 'message' => $e->getMessage()];
                Log::error('[VEVS webhook] Row failed', [
                    'rid'   => $requestId,
                    'index' => $idx,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        // 5) Response
        // 207 if some rows failed; otherwise 200.
        return response()->json([
            'ok'                => $failed === 0,
            'rid'               => $requestId,
            'import_dispatched' => $importDispatched,
            'processed'         => $processed,
            'failed'            => $failed,
            'errors'            => $errors,
        ], $failed ? 207 : 200);
    }

    /**
     * Parse JSON or form payload into array|null.
     */
    protected function parsePayload(Request $request): ?array
    {
        if ($request->isJson()) {
            $json = $request->json()->all();
            if (is_array($json) && ! empty($json)) {
                return $json;
            }
        }

        $all = $request->all();
        if (is_array($all) && ! empty($all)) {
            return $all;
        }

        $raw = (string) $request->getContent();
        if ($raw !== '') {
            try {
                $decoded = json_decode($raw, true, 512, JSON_THROW_ON_ERROR);
                if (is_array($decoded) && ! empty($decoded)) {
                    return $decoded;
                }
            } catch (\Throwable $e) {
                // ignore
            }
        }

        return null;
    }
}
