<?php
// app/Http/Controllers/Webhooks/StripeWebhookController.php

declare(strict_types=1);

namespace App\Http\Controllers\Webhooks;

use App\Http\Controllers\Controller;
use App\Models\Job;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Stripe\StripeClient;

class StripeWebhookController extends Controller
{
    /**
     * Allow Route::post('...', Ctrl::class) and ->handle()
     */
    public function __invoke(Request $request)
    {
        return $this->handle($request);
    }

    public function handle(Request $request)
    {
        // Prefer services.stripe.webhook_secret; fallback to payments.drivers.stripe.webhook_secret
        $secret  = (string) (config('services.stripe.webhook_secret')
            ?: config('payments.drivers.stripe.webhook_secret', ''));

        $payload = $request->getContent();
        $sig     = (string) $request->header('Stripe-Signature');

        try {
            $event = $secret
                ? \Stripe\Webhook::constructEvent($payload, $sig, $secret)
                : json_decode($payload);
        } catch (\Throwable $e) {
            Log::warning('Stripe signature verification failed', ['err' => $e->getMessage()]);
            return response('Invalid signature', 400);
        }

        $type   = $event->type ?? null;
        $object = $event->data->object ?? null;

        switch ($type) {
            /* ============================================================
             * SUCCESSFUL FLOWS
             * ============================================================
             */

            case 'payment_intent.succeeded': {
                /** @var \Stripe\PaymentIntent $pi */
                $pi = $object;

                // If this is a manual-capture PI and it's now succeeded, mark deposit captured/paid.
                if (($pi->capture_method ?? null) === 'manual' && ($pi->status ?? null) === 'succeeded') {
                    $this->markDepositCapturedForPaymentIntent($pi);
                }

                // Then upsert/update a normal Payment row for the charge (idempotent).
                $existing = $this->findRelatedPaymentByProviderId($pi->id)
                    ?? $this->findRelatedPaymentByLegacyCols($pi->id);

                if ($existing) {
                    $this->updatePaymentFromSucceededPI($existing, $pi);

                    if ($existing->job_id) {
                        // Optional: mark job paid
                        Job::whereKey($existing->job_id)->update(['status' => 'paid']);
                    }

                    $this->maybeSendReceipt($existing, $pi);

                    Log::info('payment_intent.succeeded (existing Payment updated)', [
                        'pi'         => $pi->id,
                        'payment_id' => $existing->id,
                        'job_id'     => $existing->job_id,
                    ]);
                    break;
                }

                // Link via Job reference or metadata and create a Payment if needed
                $job = Job::where('stripe_payment_intent', $pi->id)->first();
                if (! $job && ! empty($pi->metadata['job_id'])) {
                    $job = Job::find((int) $pi->metadata['job_id']);
                }

                if ($job) {
                    DB::transaction(function () use ($pi, $job) {
                        $chargeId = $pi->latest_charge ?? ($pi->charges->data[0]->id ?? null);

                        /** @var Payment $paymentRow */
                        $paymentRow = Payment::firstOrCreate(
                            ['reference' => $pi->id],
                            array_filter([
                                'job_id'       => $job->id,
                                'customer_id'  => $job->customer_id,
                                'amount_cents' => (int) ($pi->amount_received ?? 0),
                                'currency'     => strtoupper((string) ($pi->currency ?? ($job->currency ?? 'NZD'))),
                                'status'       => 'succeeded',
                            ]) + $this->stripeColumnsForCreate($pi, $chargeId) + $this->paidAtColumnForCreate()
                        );

                        $this->maybeBackfillProviderColumns($paymentRow, $pi);
                        $job->status = 'paid';
                        $job->save();

                        $this->maybeSendReceipt($paymentRow, $pi);
                    });

                    Log::info('payment_intent.succeeded processed (created/linked via Job)', [
                        'pi'     => $pi->id,
                        'job_id' => $job->id,
                    ]);
                } else {
                    Log::warning('PI succeeded but no matching Job or Payment found', [
                        'pi'      => $pi->id,
                        'pi_meta' => $pi->metadata ?? [],
                    ]);
                }

                break;
            }

            // Manual capture flow emits a separate event for the charge capture
            case 'charge.captured': {
                /** @var \Stripe\Charge $ch */
                $ch   = $object;
                $piId = $ch->payment_intent ?? null;

                if ($piId) {
                    $this->markDepositCapturedForPaymentIntentId($piId, $ch->id ?? null);
                }
                break;
            }

            /* ============================================================
             * FAILURE / OTHER SIGNALS
             * ============================================================
             */

            case 'payment_intent.payment_failed': {
                /** @var \Stripe\PaymentIntent $pi */
                $pi = $object;
                Log::info('payment_intent.payment_failed', ['pi' => $pi->id]);

                $payment = $this->findRelatedPaymentByProviderId($pi->id)
                    ?? $this->findRelatedPaymentByLegacyCols($pi->id);

                if ($payment) {
                    $payment->status = 'failed';
                    $payment->save();
                }
                break;
            }

            case 'payment_intent.amount_capturable_updated': {
                // Hook for auto-capture if you want it
                $this->onAmountCapturable(app(StripeClient::class), $event);
                break;
            }

            default:
                // No-op
                break;
        }

        return response()->json(['received' => true]);
    }

    /* ----------------------------------------------------------------------
     * Deposit capture helpers (schema-aware & idempotent)
     * ---------------------------------------------------------------------- */

    /**
     * Mark the booking deposit captured/paid for a given PaymentIntent object.
     * Search order: metadata.deposit_id → saved PI id on deposit → metadata.job_id (latest deposit).
     */
    protected function markDepositCapturedForPaymentIntent(\Stripe\PaymentIntent $pi): void
    {
        $deposit = $this->findDepositForPI([
            'id'       => $pi->id,
            'metadata' => (array) ($pi->metadata ?? []),
        ]);

        if ($deposit) {
            $latestCharge = $pi->latest_charge ?? ($pi->charges->data[0]->id ?? null);
            $this->applyDepositCapturedState($deposit, $pi->id, $latestCharge);
        }
    }

    /**
     * Mark the booking deposit captured/paid by PaymentIntent id (for charge.captured).
     */
    protected function markDepositCapturedForPaymentIntentId(string $piId, ?string $chargeId = null): void
    {
        $deposit = Payment::query()
            ->where('type', 'booking_deposit')
            ->when(Schema::hasColumn('payments', 'stripe_payment_intent_id'),
                fn($q) => $q->where('stripe_payment_intent_id', $piId),
                fn($q) => $q->where('stripe_payment_intent', $piId) // legacy
            )
            ->latest('id')
            ->first();

        if ($deposit) {
            $this->applyDepositCapturedState($deposit, $piId, $chargeId);
        }
    }

    /**
     * Finder used by both succeeded and charge.captured paths.
     */
    protected function findDepositForPI(array $pi): ?Payment
    {
        $depositId = Arr::get($pi, 'metadata.deposit_id');
        if ($depositId) {
            $dep = Payment::find((int) $depositId);
            if ($dep && $dep->type === 'booking_deposit') {
                return $dep;
            }
        }

        $piId = Arr::get($pi, 'id');
        if ($piId) {
            $dep = Payment::query()
                ->where('type', 'booking_deposit')
                ->when(Schema::hasColumn('payments', 'stripe_payment_intent_id'),
                    fn($q) => $q->where('stripe_payment_intent_id', $piId),
                    fn($q) => $q->where('stripe_payment_intent', $piId) // legacy
                )
                ->latest('id')
                ->first();
            if ($dep) return $dep;
        }

        $jobId = (int) Arr::get($pi, 'metadata.job_id', 0);
        if ($jobId > 0) {
            return Payment::query()
                ->where('type', 'booking_deposit')
                ->where('job_id', $jobId)
                ->latest('id')
                ->first();
        }

        return null;
    }

    /**
     * Apply captured/paid state to a deposit Payment (idempotent & schema-aware).
     */
    protected function applyDepositCapturedState(Payment $deposit, ?string $piId, ?string $chargeId): void
    {
        // Persist IDs if empty
        if ($piId) {
            if (Schema::hasColumn('payments', 'stripe_payment_intent_id') && empty($deposit->stripe_payment_intent_id)) {
                $deposit->stripe_payment_intent_id = $piId;
            }
            if (Schema::hasColumn('payments', 'stripe_payment_intent') && empty($deposit->stripe_payment_intent)) {
                $deposit->stripe_payment_intent = $piId;
            }
        }

        if ($chargeId) {
            if (Schema::hasColumn('payments', 'stripe_charge_id') && empty($deposit->stripe_charge_id)) {
                $deposit->stripe_charge_id = $chargeId;
            }
            if (Schema::hasColumn('payments', 'stripe_charge') && empty($deposit->stripe_charge)) {
                $deposit->stripe_charge = $chargeId;
            }
        }

        // Mark as paid/captured
        $deposit->status = 'succeeded'; // or 'captured' internally

        if (Schema::hasColumn('payments', 'deposit_confirmed_at') && empty($deposit->deposit_confirmed_at)) {
            $deposit->deposit_confirmed_at = now();
        }
        if (Schema::hasColumn('payments', 'captured_at') && empty($deposit->captured_at)) {
            $deposit->captured_at = now();
        }
        if (Schema::hasColumn('payments', 'paid_at') && empty($deposit->paid_at)) {
            $deposit->paid_at = now();
        }

        // Optional provider backfill
        $this->maybeBackfillProviderColumns($deposit, (object) ['id' => $piId]);

        $deposit->save();

        Log::info('Booking deposit marked succeeded/captured', [
            'payment_id' => $deposit->id,
            'pi'         => $piId,
            'charge'     => $chargeId,
        ]);
    }

    /* ----------------------------------------------------------------------
     * Helpers: Payment linking / updating
     * ---------------------------------------------------------------------- */

    protected function findRelatedPaymentByProviderId(string $piId): ?Payment
    {
        $query = Payment::query();

        $hasProviderId = Schema::hasColumn('payments', 'provider_id');
        $hasGatewayId  = Schema::hasColumn('payments', 'gateway_id');

        if (! $hasProviderId && ! $hasGatewayId) {
            return null;
        }

        if ($hasProviderId) {
            $query->orWhere('provider_id', $piId);
        }
        if ($hasGatewayId) {
            $query->orWhere('gateway_id', $piId);
        }

        return $query->first();
    }

    protected function findRelatedPaymentByLegacyCols(string $piId): ?Payment
    {
        return Payment::query()
            ->when(Schema::hasColumn('payments', 'stripe_payment_intent_id'),
                fn($q) => $q->orWhere('stripe_payment_intent_id', $piId))
            ->when(Schema::hasColumn('payments', 'stripe_payment_intent'),
                fn($q) => $q->orWhere('stripe_payment_intent', $piId))
            ->orWhere('reference', $piId)
            ->first();
    }

    protected function updatePaymentFromSucceededPI(Payment $payment, \Stripe\PaymentIntent $pi): void
    {
        $payment->status       = 'succeeded';
        $payment->amount_cents = $payment->amount_cents ?: (int) ($pi->amount_received ?? 0);

        if (empty($payment->currency)) {
            $payment->currency = strtoupper((string) ($pi->currency ?? 'NZD'));
        }

        if (Schema::hasColumn('payments', 'paid_at') && empty($payment->paid_at)) {
            $payment->paid_at = now();
        }

        $this->maybeBackfillProviderColumns($payment, $pi);

        $latestCharge = $pi->latest_charge ?? ($pi->charges->data[0]->id ?? null);
        if ($latestCharge) {
            if (Schema::hasColumn('payments', 'stripe_charge') && empty($payment->stripe_charge)) {
                $payment->stripe_charge = $latestCharge;
            }
            if (Schema::hasColumn('payments', 'stripe_charge_id') && empty($payment->stripe_charge_id)) {
                $payment->stripe_charge_id = $latestCharge;
            }
        }

        if (Schema::hasColumn('payments', 'stripe_payment_intent') && empty($payment->stripe_payment_intent)) {
            $payment->stripe_payment_intent = $pi->id;
        }
        if (Schema::hasColumn('payments', 'stripe_payment_intent_id') && empty($payment->stripe_payment_intent_id)) {
            $payment->stripe_payment_intent_id = $pi->id;
        }

        $payment->save();
    }

    protected function maybeBackfillProviderColumns(Payment $payment, object $piLike): void
    {
        if (Schema::hasColumn('payments', 'provider') && empty($payment->provider)) {
            $payment->provider = 'stripe';
        }

        $piId = $piLike->id ?? null;
        if ($piId) {
            if (Schema::hasColumn('payments', 'provider_id') && empty($payment->provider_id)) {
                $payment->provider_id = $piId;
            }
            if (Schema::hasColumn('payments', 'gateway_id') && empty($payment->gateway_id)) {
                $payment->gateway_id = $piId;
            }
        }
    }

    protected function stripeColumnsForCreate(\Stripe\PaymentIntent $pi, ?string $chargeId = null): array
    {
        $cols = [];

        if (Schema::hasColumn('payments', 'stripe_payment_intent')) {
            $cols['stripe_payment_intent'] = $pi->id;
        }
        if (Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
            $cols['stripe_payment_intent_id'] = $pi->id;
        }
        if ($chargeId && Schema::hasColumn('payments', 'stripe_charge')) {
            $cols['stripe_charge'] = $chargeId;
        }
        if ($chargeId && Schema::hasColumn('payments', 'stripe_charge_id')) {
            $cols['stripe_charge_id'] = $chargeId;
        }

        return $cols;
    }

    protected function paidAtColumnForCreate(): array
    {
        return Schema::hasColumn('payments', 'paid_at') ? ['paid_at' => now()] : [];
    }

    /* ----------------------------------------------------------------------
     * Optional hooks
     * ---------------------------------------------------------------------- */

    protected function onAmountCapturable(StripeClient $stripe, object $event): void
    {
        // Example: auto-capture here if desired
        // $pi = $event->data->object;
        // $stripe->paymentIntents->capture($pi->id, [...]);
    }

    protected function maybeSendReceipt(Payment $payment, \Stripe\PaymentIntent $pi): void
    {
        // Implement notification if required
    }
}
