<?php

namespace App\Jobs;

use App\Models\Job as HoldJob; // <- your Job model
use App\Models\Vehicle;
use App\Services\VevsClient;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;

class ImportVevsReservation implements ShouldQueue
{
    use Dispatchable, Queueable;

    public function __construct(public string $refId)
    {
    }

    public function handle(): void
    {
        $reservation = VevsClient::make()->getReservation($this->refId);

        // --- Resolve Vehicle from VEVS car_id ---------------------------------
        $carId   = $reservation['car_id'] ?? null;
        $vehicle = null;

        if (! empty($carId)) {
            $vehicle = Vehicle::where('vevs_vehicle_id', (string) $carId)->first();
        }

        // --- Build attribute payload for the Job -------------------------------
        $attributes = [
            'pickup_at'           => $reservation['from'] ?? null,
            'return_at'           => $reservation['to'] ?? null,
            'pickup_location_id'  => $reservation['pickup_id'] ?? null,
            'return_location_id'  => $reservation['return_id'] ?? null,
            'total_price'         => $reservation['total_price'] ?? null,
            'required_deposit'    => $reservation['required_deposit'] ?? null,
            'security_deposit'    => $reservation['security_deposit'] ?? null,
            'customer_name'       => $reservation['c_driver_name'] ?? $reservation['c_name'] ?? null,
            'customer_email'      => $reservation['c_email'] ?? null,
            'customer_phone'      => $reservation['c_phone'] ?? null,
            // set flow/status defaults if missing (left to your schema)
        ];

        // If we resolved a Vehicle, attach it via the foreign key
        if ($vehicle) {
            $attributes['vehicle_id'] = $vehicle->id;
        } else {
            // Optionally leave vehicle_id untouched on existing jobs by NOT setting it here.
            // If you prefer to explicitly null it when no match is found, keep this line:
            $attributes['vehicle_id'] = null;
        }

        // --- Upsert the Job record --------------------------------------------
        HoldJob::updateOrCreate(
            [
                'booking_reference' => $reservation['ref_id']
                    ?? $reservation['booking_id']
                    ?? null,
            ],
            $attributes
        );
    }
}
