<?php

declare(strict_types=1);

namespace App\Mail;

use App\Models\Owner;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Carbon;

class OwnerStatementMail extends Mailable
{
    use Queueable;
    use SerializesModels;

    public Owner $owner;
    public Carbon $fromDate;
    public Carbon $toDate;
    public array $totals;
    public int $payoutAmountCents;
    public array $previewTotals;
    public Carbon $previewPayoutDate;
    public string $statementId;
    public $vehicles; // Collection

    protected string $pdfData;
    protected string $pdfFileName;

    public function __construct(
        Owner $owner,
        Carbon $fromDate,
        Carbon $toDate,
        array $totals,
        int $payoutAmountCents,
        $vehicles,
        string $statementId,
        array $previewTotals,
        Carbon $previewPayoutDate,
        string $pdfData,
        string $pdfFileName,
    ) {
        $this->owner             = $owner;
        $this->fromDate          = $fromDate;
        $this->toDate            = $toDate;
        $this->totals            = $totals;
        $this->payoutAmountCents = $payoutAmountCents;
        $this->vehicles          = $vehicles;
        $this->statementId       = $statementId;
        $this->previewTotals     = $previewTotals;
        $this->previewPayoutDate = $previewPayoutDate;
        $this->pdfData           = $pdfData;
        $this->pdfFileName       = $pdfFileName;
    }

    public function build(): self
    {
        $subject = sprintf(
            'Owner Statement %s – %s',
            $this->fromDate->toDateString(),
            $this->toDate->toDateString(),
        );

        $fmt = fn (int $cents): string => '$' . number_format($cents / 100, 2);

        $payoutReference = $this->owner->bank_reference
            ?: ('DD-' . $this->statementId);

        return $this->subject($subject)
            ->view('mail.owner_statement')
            ->with([
                'owner'             => $this->owner,
                'from'              => $this->fromDate,
                'to'                => $this->toDate,
                'totals'            => $this->totals,
                'payoutAmountCents' => $this->payoutAmountCents,
                'vehicles'          => $this->vehicles,
                'statementId'       => $this->statementId,
                'payoutReference'   => $payoutReference,
                'previewTotals'     => $this->previewTotals,
                'previewPayoutDate' => $this->previewPayoutDate,
                'fmt'               => $fmt,
                'previewJobs'       => $this->previewJobs,
                
            ])
            ->attachData(
                data: $this->pdfData,
                name: $this->pdfFileName,
                options: ['mime' => 'application/pdf'],
            );
    }
}
