<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Customer extends Authenticatable
{
    use HasFactory;
    use Notifiable;

    /** @var string[] */
    protected $fillable = [
        'first_name',
        'last_name',
        'email',
        'phone',

        // Payments
        'default_payment_method_id',
        'stripe_customer_id',

        // Address
        'address_line1',
        'address_line2',
        'address_city',
        'address_region',
        'address_postcode',
        'address_country',

        // Portal-related (not all columns exist on this env – see helpers below)
        'portal_token',
        'login_token',
        'login_token_expires_at',
        'portal_last_login_at',
        'portal_last_seen_at',
        'portal_timezone',
        'portal_magic_redirect',

        // Freeform notes/metadata
        'meta',
    ];

    /** @var array<string,string> */
    protected $casts = [
        'meta'                   => 'array',
        'login_token_expires_at' => 'datetime',
        'portal_last_login_at'   => 'datetime',
        'portal_last_seen_at'    => 'datetime',
    ];

    /** @var string[] */
    protected $hidden = [
        'login_token',
        'stripe_customer_id',
        'default_payment_method_id',
        'meta',
        // 'portal_token',
    ];

    /** @var string[] */
    protected $appends = [
        'name',
    ];

    // ---------------------------------------------------------------------
    // Auth integration (passwordless)
    // ---------------------------------------------------------------------

    public function getAuthPassword(): string
    {
        // No password used for portal auth (magic-links only)
        return '';
    }

    public function getRememberToken(): ?string
    {
        return null;
    }

    public function setRememberToken($value): void
    {
        // no-op
    }

    public function getRememberTokenName(): string
    {
        return '';
    }

    public function routeNotificationForMail(): ?string
    {
        return $this->email;
    }

    // ---------------------------------------------------------------------
    // Relationships
    // ---------------------------------------------------------------------

    public function bookings(): HasMany
    {
        return $this->hasMany(Booking::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function deposits(): HasMany
    {
        return $this->hasMany(Deposit::class);
    }

    // ---------------------------------------------------------------------
    // Scopes & finders
    // ---------------------------------------------------------------------

    /** Case-insensitive email lookup */
    public function scopeWhereEmail(Builder $query, string $email): Builder
    {
        return $query->whereRaw('LOWER(email) = ?', [mb_strtolower(trim($email))]);
    }

    public static function findByEmail(string $email): ?self
    {
        return static::whereEmail($email)->first();
    }

    /**
     * NOTE: portal_token is currently NOT stored in the DB on this env.
     * Keep the scope for future, but make it a no-op for now.
     */
    public function scopeWherePortalToken(Builder $query, string $token): Builder
    {
        return $query->whereRaw('1 = 0');
    }

    public static function findByPortalToken(string $token): ?self
    {
        return static::wherePortalToken($token)->first();
    }

    // ---------------------------------------------------------------------
    // Mutators
    // ---------------------------------------------------------------------

    public function setEmailAttribute(?string $value): void
    {
        $this->attributes['email'] = $value ? mb_strtolower(trim($value)) : null;
    }

    // ---------------------------------------------------------------------
    // Accessors
    // ---------------------------------------------------------------------

    /** Full name; falls back sensibly */
    public function getNameAttribute(): string
    {
        $first = $this->first_name ?? '';
        $last  = $this->last_name ?? '';
        $full  = trim($first . ' ' . $last);

        if ($full !== '') {
            return $full;
        }

        return (string)($this->attributes['name'] ?? ($this->email ?? ''));
    }

    /** A short “Adam C.” style label */
    public function getShortNameAttribute(): string
    {
        $first = trim((string) $this->first_name);
        $last  = trim((string) $this->last_name);

        if ($first !== '' && $last !== '') {
            return $first . ' ' . mb_substr($last, 0, 1) . '.';
        }

        return $this->name;
    }

    // ---------------------------------------------------------------------
    // Portal (magic-link) helpers – stateless for now
    // ---------------------------------------------------------------------

    /**
     * Long-lived portal token used in links.
     * Currently NOT stored in DB – we generate a deterministic token
     * so links are stable without any saves.
     */
    public function ensurePortalToken(int $length = 40): string
    {
        $id = $this->getKey();

        if (! $id) {
            // Unsaved model – just return an ephemeral random token
            return Str::random($length);
        }

        $base = 'customer-'.$id.'-'.config('app.key');

        return substr(hash('sha256', $base), 0, $length);
    }

    /**
     * "Rotate" portal token – without persistence this just returns a
     * new random token. Once you add a real column, you can switch this
     * back to saving a random value.
     */
    public function rotatePortalToken(int $length = 40): string
    {
        return Str::random($length);
    }

    /**
     * Issue a fresh one-time login token.
     *
     * TEMP: we cannot store login_token/login_token_expires_at yet
     * (columns missing), so this simply returns a random opaque token
     * without persisting it.
     */
    public function issueLoginToken($expiresIn = null): string
    {
        return Str::random(64);
    }

    /**
     * Validate a presented token.
     *
     * TEMP: without stored hashes, we must always return false so that
     * magic-link logins are effectively disabled (but the admin UI can
     * still generate URLs without errors).
     */
    public function consumeLoginToken(string $rawToken): bool
    {
        return false;
    }

    /** Clear any outstanding magic link token – no-op until we persist them. */
    public function clearLoginToken(): void
    {
        // no-op
    }

    /** Mark a heartbeat/last seen – no-op until we add portal_last_seen_at column. */
    public function touchPortalSeen(): void
    {
        // no-op for now
    }

    /**
     * Create a full magic-link URL for the controller to send.
     * Guarded so it doesn't crash if the route doesn't exist.
     */
    public function buildMagicLoginUrl(string $routeName, array $params = []): string
    {
        if (! Route::has($routeName)) {
            // Route not available on this env – return harmless placeholder.
            return '#';
        }

        $raw     = $this->issueLoginToken();
        $payload = array_merge(['token' => $raw, 'email' => $this->email], $params);

        return route($routeName, $payload);
    }
}
