<?php

declare(strict_types=1);

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Damage extends Model
{
    use HasFactory;

    protected $fillable = [
        'vehicle_id',
        'job_id',
        'reported_by_user_id',
        'found_at',
        'location',
        'type',
        'severity',
        'estimated_cost_cents',
        'requires_repair',
        'repair_status',
        'is_preexisting',
        'photos',
        'notes',
    ];

    protected $casts = [
        'found_at'          => 'datetime',
        'requires_repair'   => 'boolean',
        'is_preexisting'    => 'boolean',
        'photos'            => 'array',
        'estimated_cost_cents' => 'integer',
    ];

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    public function reportedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'reported_by_user_id');
    }

    public function getEstimatedCostDollarsAttribute(): ?string
    {
        if ($this->estimated_cost_cents === null) {
            return null;
        }

        return number_format($this->estimated_cost_cents / 100, 2);
    }
}
