<?php

declare(strict_types=1);

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\Event as LaravelEvent;
use Illuminate\Support\Facades\Schema;

/**
 * Canonical deposits schema (this model assumes these columns exist):
 * - job_id, customer_id (nullable)
 * - authorized_cents (int), captured_cents (int default 0)
 * - currency (3)
 * - reference (nullable)
 * - stripe_payment_intent (nullable), stripe_charge (nullable), stripe_payment_method (nullable)
 * - status: authorized|captured|released|canceled|failed
 * - last_reason (nullable), meta (json nullable)
 * - authorized_at (nullable datetime)
 * - planned_cancel_at (nullable datetime)
 * - released_at (nullable datetime)
 * - canceled_at (nullable datetime)
 */
class Deposit extends Model
{
    /** @var string[] */
    public const STATUSES = ['authorized', 'authorised', 'captured', 'released', 'canceled', 'failed'];

    /* ---------------------------------------------------------------------
     | Mass assignment
     |--------------------------------------------------------------------- */
    protected $fillable = [
        // Ownership
        'job_id',
        'customer_id',

        // Money (cents)
        'authorized_cents',
        'captured_cents',
        'currency',

        // References
        'reference',
        'stripe_payment_intent',
        'stripe_charge',
        'stripe_payment_method',

        // Status / reasons
        'status',
        'last_reason',

        // Metadata
        'meta',

        // Lifecycle timestamps
        'authorized_at',
        'planned_cancel_at',
        'released_at',
        'canceled_at',
    ];

    /* ---------------------------------------------------------------------
     | Casting
     |--------------------------------------------------------------------- */
    protected $casts = [
        'authorized_cents'   => 'integer',
        'captured_cents'     => 'integer',
        'meta'               => 'array',

        'authorized_at'      => 'datetime',
        'planned_cancel_at'  => 'datetime',
        'released_at'        => 'datetime',
        'canceled_at'        => 'datetime',
    ];

    /* ---------------------------------------------------------------------
     | Default attributes
     |--------------------------------------------------------------------- */
    protected $attributes = [
        'authorized_cents' => 0,
        'captured_cents'   => 0,
        'status'           => 'authorized',
        'currency'         => 'NZD',
    ];

    /* ---------------------------------------------------------------------
     | Relationships
     |--------------------------------------------------------------------- */

    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    public function customer(): BelongsTo
    {
        // Optional — only if you have a Customer model
        return $this->belongsTo(Customer::class);
    }

    /* ---------------------------------------------------------------------
     | Query scopes
     |--------------------------------------------------------------------- */

    /** Deposits that are in an "authorized" state (AU/US spelling). */
    public function scopeAuthorized($q)
    {
        return $q->whereIn('status', ['authorized', 'authorised']);
    }

    /**
     * "Active" holds we might still act on:
     *   - still authorized or partially captured
     *   - not released/canceled
     */
    public function scopeActive($q)
    {
        return $q
            ->whereIn('status', ['authorized', 'authorised', 'captured'])
            ->whereNull('released_at')
            ->whereNull('canceled_at');
    }

    /** Still has remaining amount to capture. */
    public function scopePendingCapture($q)
    {
        return $q->active()->whereColumn('authorized_cents', '>', 'captured_cents');
    }

    /* ---------------------------------------------------------------------
     | State helpers
     |--------------------------------------------------------------------- */

    public function isAuthorized(): bool
    {
        return in_array($this->status, ['authorized', 'authorised'], true);
    }

    public function isCaptured(): bool
    {
        return $this->status === 'captured';
    }

    public function isReleased(): bool
    {
        return $this->status === 'released' || !is_null($this->released_at);
    }

    public function isCanceled(): bool
    {
        return $this->status === 'canceled' || !is_null($this->canceled_at);
    }

    /* ---------------------------------------------------------------------
     | Display helpers
     |--------------------------------------------------------------------- */

    public function displayName(): string
    {
        if ($this->relationLoaded('customer') || $this->customer) {
            return $this->customer->name
                ?? $this->customer->email
                ?? "Deposit #{$this->id}";
        }
        return "Deposit #{$this->id}";
    }

    /* ---------------------------------------------------------------------
     | Amount helpers
     |--------------------------------------------------------------------- */

    /** Remaining amount (in cents) that can still be captured. */
    public function remainingHoldCents(): int
    {
        $auth = (int) ($this->authorized_cents ?? 0);
        $cap  = (int) ($this->captured_cents   ?? 0);
        return max(0, $auth - $cap);
    }

    /** Back-compat alias. */
    public function remainingCents(): int
    {
        return $this->remainingHoldCents();
    }

    public function authorizedDollars(): string
    {
        return number_format(((int) ($this->authorized_cents ?? 0)) / 100, 2);
    }

    public function capturedDollars(): string
    {
        return number_format(((int) ($this->captured_cents ?? 0)) / 100, 2);
    }

    /* ---------------------------------------------------------------------
     | Planning helpers
     |--------------------------------------------------------------------- */

    /**
     * If planned_cancel_at is missing, compute and set a sensible default based on:
     *  - the stored value if present
     *  - else job->end_at + config('holds.default_cancel_after_minutes')
     *  - additionally, cap by Stripe auth window (safeguard) if configured
     */
    public function ensurePlannedCancelAt(): void
    {
        if ($this->planned_cancel_at) {
            return;
        }

        $minutes = (int) (config('holds.default_cancel_after_minutes') ?? 720);
        $candidate = null;

        if ($this->relationLoaded('job') || $this->job) {
            $end = optional($this->job->end_at)->copy();
            if ($end) {
                $candidate = $end->clone()->addMinutes($minutes);
            }
        }

        // Fallback: from authorized_at if no job end present
        if (!$candidate && $this->authorized_at) {
            $candidate = $this->authorized_at->clone()->addMinutes($minutes);
        }

        // Stripe auth window safeguard
        $days = (int) (config('holds.stripe_auth_window_days') ?? 7);
        if ($days > 0 && $this->authorized_at && $candidate) {
            $cap = $this->authorized_at->clone()->addDays($days);
            if ($candidate->greaterThan($cap)) {
                $candidate = $cap;
            }
        }

        if ($candidate) {
            $this->planned_cancel_at = $candidate;
            $this->save();
        }
    }

    /* ---------------------------------------------------------------------
     | Stripe actions
     |--------------------------------------------------------------------- */

    /**
     * Capture part or all of the hold (PaymentIntent preferred; fallback to Charge).
     *
     * @throws \Symfony\Component\HttpKernel\Exception\HttpException if invalid state
     */
    public function capture(\Stripe\StripeClient $stripe, int $amountToCapture, ?string $reason = null): self
    {
        abort_unless($amountToCapture > 0, 422, 'Capture amount must be > 0');
        abort_if(!$this->isAuthorized() && !$this->isCaptured(), 422, 'Deposit not in an authorizable state');

        $remaining = $this->remainingHoldCents();
        abort_unless($remaining > 0, 422, 'Nothing remaining to capture.');
        abort_unless($amountToCapture <= $remaining, 422, 'Capture exceeds remaining hold.');

        if (!empty($this->stripe_payment_intent)) {
            $stripe->paymentIntents->capture($this->stripe_payment_intent, [
                'amount_to_capture' => $amountToCapture,
            ]);
        } elseif (!empty($this->stripe_charge)) {
            $stripe->charges->capture($this->stripe_charge, [
                'amount' => $amountToCapture,
            ]);
        } else {
            abort(422, 'No Stripe reference present to capture.');
        }

        // Update local state
        $this->captured_cents = (int) $this->captured_cents + $amountToCapture;
        $this->last_reason    = $reason;
        $this->status         = ($this->captured_cents >= (int) $this->authorized_cents)
            ? 'captured'
            : 'authorized';
        $this->save();

        // Mirror to Payments table if it exists (safely check columns that may vary by install)
        if (class_exists(\App\Models\Payment::class)) {
            $payload = [
                'job_id'                => $this->job_id,
                'amount_cents'          => $amountToCapture,
                'currency'              => $this->currency ?? 'NZD',
                'status'                => 'succeeded',
                'type'                  => 'capture',
                'provider'              => 'stripe',
                'stripe_payment_intent' => $this->stripe_payment_intent,
                'stripe_charge'         => $this->stripe_charge,
                'reference'             => $this->stripe_charge ?: $this->stripe_payment_intent,
                'meta'                  => ['reason' => $reason, 'deposit_id' => $this->id],
            ];

            // Add paid_at when present in schema
            if (Schema::hasColumn('payments', 'paid_at')) {
                $payload['paid_at'] = now();
            }

            /** @var \App\Models\Payment $payment */
            \App\Models\Payment::create($payload);
        }

        // Optional event log (if your app has a generic Event model)
        if (class_exists(\App\Models\Event::class)) {
            \App\Models\Event::create([
                'job_id'  => $this->job_id,
                'type'    => 'deposit.captured',
                'message' => 'Captured ' . number_format($amountToCapture / 100, 2) . ' ' . ($this->currency ?? 'NZD') . ' from hold',
                'meta'    => ['deposit_id' => $this->id, 'reason' => $reason],
            ]);
        }

        // Fire a Laravel event if you prefer listeners
        if (class_exists(\App\Events\DepositCaptured::class)) {
            LaravelEvent::dispatch(new \App\Events\DepositCaptured($this, $amountToCapture, $reason));
        }

        return $this;
    }

    /**
     * Mark the hold as released (Stripe cancellation handled elsewhere).
     * Handy for your AutoCancel job once Stripe confirms the release.
     */
    public function markReleased(?string $reason = null): self
    {
        $this->status        = 'released';
        $this->released_at   = now();
        $this->last_reason   = $reason ?? $this->last_reason;
        $this->save();

        if (class_exists(\App\Models\Event::class)) {
            \App\Models\Event::create([
                'job_id'  => $this->job_id,
                'type'    => 'deposit.released',
                'message' => 'Hold released',
                'meta'    => ['deposit_id' => $this->id, 'reason' => $this->last_reason],
            ]);
        }

        if (class_exists(\App\Events\DepositReleased::class)) {
            LaravelEvent::dispatch(new \App\Events\DepositReleased($this, $reason));
        }

        return $this;
    }

    /**
     * Mark the hold as canceled (e.g., manual cancel or failure to auth).
     */
    public function markCanceled(?string $reason = null): self
    {
        $this->status        = 'canceled';
        $this->canceled_at   = now();
        $this->last_reason   = $reason ?? $this->last_reason;
        $this->save();

        if (class_exists(\App\Models\Event::class)) {
            \App\Models\Event::create([
                'job_id'  => $this->job_id,
                'type'    => 'deposit.canceled',
                'message' => 'Hold canceled',
                'meta'    => ['deposit_id' => $this->id, 'reason' => $this->last_reason],
            ]);
        }

        if (class_exists(\App\Events\DepositCanceled::class)) {
            LaravelEvent::dispatch(new \App\Events\DepositCanceled($this, $reason));
        }

        return $this;
    }
}
