<?php

declare(strict_types=1);

namespace App\Models;

use App\Console\Commands\AutoCancelHolds;
use App\Enums\PaymentStatus;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasManyThrough;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\URL;

class Job extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'jobs';

    /**
     * Limit array output to fields we actually want to expose.
     */
    public function toArray(): array
    {
        $base = parent::toArray();

        return Arr::only($base, [
            'id',
            'reference',
            'external_reference',
            'customer_name',
            'customer_email',
            'customer_phone',
            'status',
            'charge_amount_cents',
            'hold_amount_cents',
            'captured_amount_cents',
            'authorized_amount_cents',
            'start_at',
            'end_at',
            'actual_completed_at',
            'created_at',
            'updated_at',
            'paid_amount_cents',
            'remaining_amount_cents',
            'hold_amount_cents_from_flow',
            'display_ref',
            'is_paid_in_full',
            'balance_due_cents',
            'vehicle_label',
        ]);
    }

    protected $hidden = [
        'meta',
        'comms_log',
        'billing_address',
        'external_payload',
    ];

    protected $fillable = [
        'flow_id',
        'brand_id',
        'booking_id',
        'vehicle_id',
        'created_by',
        'external_reference',
        'reference',
        'customer_name',
        'customer_email',
        'customer_phone',
        'billing_address',
        'start_at',
        'end_at',
        'actual_completed_at',
        'charge_amount_cents',
        'hold_amount_cents',
        'captured_amount_cents',
        'authorized_amount_cents',
        'psp',
        'psp_authorization_id',
        'psp_payment_method_id',
        'psp_customer_id',
        'status',
        'meta',
        'comms_log',
        'updated_by',
        'agreement_path',
        'paid_amount_cents',
        'remaining_amount_cents',
        'balance_due_cents',
        'vevs_car_id',

        // Booking / source info
        'source_system',
        'pickup_location_id',
        'return_location_id',
        'rental_days',
        'rental_hours',

        // Money buckets (all cents)
        'base_rental_cents',
        'extra_mileage_fee_cents',
        'extras_cents',
        'other_fees_cents',
        'insurance_cents',
        'vat_cents',
        'total_price_cents',
        'currency_code',

        // Extended pricing from VEVS or our own engine
        'tax_cents',
        'vat_rate',
        'total_before_discount_cents',
        'required_deposit_cents',
        'security_deposit_cents',
        'payment_method_external',
        'external_status',

        // Extended customer / drivers / travel
        'primary_driver_dob',
        'primary_driver_licence',
        'secondary_driver_name',
        'secondary_driver_dob',
        'secondary_driver_licence',
        'flight_number',
        'airline',
        'customer_notes',

        // External payload blob (e.g., raw VEVS payload)
        'external_payload',
    ];

    protected $casts = [
    // datetimes
    'start_at'                => 'datetime',
    'end_at'                  => 'datetime',
    'actual_completed_at'     => 'datetime',

    // customer / drivers
    'primary_driver_dob'      => 'date',
    'secondary_driver_dob'    => 'date',

    // arrays / JSON
    'billing_address'         => 'array',
    'meta'                    => 'array',
    'comms_log'               => 'array',
    'external_payload'        => 'array',

    // money / integer fields
    'charge_amount_cents'         => 'integer',
    'hold_amount_cents'           => 'integer',
    'captured_amount_cents'       => 'integer',
    'authorized_amount_cents'     => 'integer',
    'paid_amount_cents'           => 'integer',
    'remaining_amount_cents'      => 'integer',
    'balance_due_cents'           => 'integer',
    'required_deposit_cents'      => 'integer',
    'security_deposit_cents'      => 'integer',
    'tax_cents'                   => 'integer',
    'total_before_discount_cents' => 'integer',

    // vehicle / external references
    'psp_customer_id'         => 'string',
    'vevs_car_id'             => 'string',
    'pickup_location_id'      => 'integer',
    'return_location_id'      => 'integer',

    // money buckets
    'base_rental_cents'        => 'integer',
    'extra_mileage_fee_cents'  => 'integer',
    'extras_cents'             => 'integer',
    'other_fees_cents'         => 'integer',
    'insurance_cents'          => 'integer',
    'vat_cents'                => 'integer',
    'total_price_cents'        => 'integer',
    'currency_code'            => 'string',

    // VAT
    'vat_rate'                => 'float',

    // booking attributes
    'rental_days'             => 'integer',
    'rental_hours'            => 'integer',
];


    protected $appends = [
        'paid_amount_cents',
        'remaining_amount_cents',
        'hold_amount_cents_from_flow',
        'display_ref',
        'is_paid_in_full',
        'balance_due_cents',
        'vehicle_label',
    ];

    /* Relationships */

    public function flow(): BelongsTo
    {
        return $this->belongsTo(Flow::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    /**
     * Booking via external_reference → bookings.reference
     */
    public function booking(): HasOne
    {
        // external_reference on jobs → reference on bookings
        return $this->hasOne(Booking::class, 'reference', 'external_reference');
    }

    /**
     * Vehicle relationship (jobs.vehicle_id → vehicles.id)
     */
    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * Convenience: consignment Owner for this job (via Vehicle), or null.
     */
    public function owner(): ?Owner
    {
        return $this->vehicle?->owner;
    }

    /**
     * True if this job’s vehicle is a consignment vehicle.
     */
    public function hasConsignmentOwner(): bool
    {
        return $this->vehicle?->isConsignment() ?? false;
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function damages(): HasMany
    {
        return $this->hasMany(Damage::class);
    }

    /** Added: fixes "undefined relationship [customer]" */
    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class);
    }

    public function deposits(): HasMany
    {
        return $this->hasMany(Deposit::class);
    }

    public function events(): HasMany
    {
        return $this->hasMany(JobEvent::class);
    }

    public function communications(): HasMany
    {
        return $this->hasMany(Communication::class);
    }

    public function communicationEvents(): HasManyThrough
    {
        return $this->hasManyThrough(CommunicationEvent::class, Communication::class);
    }

    public function tasks(): HasMany
    {
        return $this->hasMany(JobTask::class);
    }

    public function activities(): HasManyThrough
    {
        return $this->hasManyThrough(Activity::class, JobEvent::class);
    }

    /* Scopes */

    /**
     * Common scope for "active" jobs that appear on calendar.
     *
     * Usage: Job::forCalendar()->get();
     */
    public function scopeForCalendar($query)
    {
        return $query->whereIn('status', ['confirmed', 'in_progress']);
    }

    public function scopeUnpaid($q)
    {
        return $q->where(function ($q) {
            $q->where('balance_due_cents', '>', 0)
                ->orWhereNull('is_paid')
                ->orWhere('is_paid', false);
        });
    }

    /* Aggregates / Money helpers */

    /**
     * Accessor: paid_amount_cents (prefers stored column when available).
     */
    public function getPaidAmountCentsAttribute(): int
    {
        if (
            Schema::hasColumn($this->getTable(), 'paid_amount_cents') &&
            ! is_null($this->getAttributeFromArray('paid_amount_cents'))
        ) {
            return (int) $this->getAttributeFromArray('paid_amount_cents');
        }

        return (int) $this->payments()
            ->whereIn('status', [PaymentStatus::Succeeded->value, PaymentStatus::Captured->value])
            ->sum('amount_cents');
    }

    /**
     * Helper (prefer eager-loaded aggregate if present).
     */
    public function paidAmountCents(): int
    {
        $preloaded = $this->getAttribute('paid_sum_cents');
        if ($preloaded !== null) {
            return (int) $preloaded;
        }

        return (int) $this->payments()
            ->whereIn('status', [PaymentStatus::Succeeded->value, PaymentStatus::Captured->value])
            ->sum('amount_cents');
    }

    public function owingAmountCents(): int
    {
        $charge = (int) ($this->charge_amount_cents ?? 0);

        return max(0, $charge - $this->paidAmountCents());
    }

    /** NZ$ prefix, direct sum from payments() */
    public function paidAmountFormatted(): string
    {
        $cents = (int) $this->payments()
            ->whereIn('status', [PaymentStatus::Succeeded->value, PaymentStatus::Captured->value])
            ->sum('amount_cents');

        return 'NZ$ ' . number_format($cents / 100, 2);
    }

    /** NZ$ prefix, direct sum from payments() */
    public function owingAmountFormatted(): string
    {
        $charge = (int) ($this->charge_amount_cents ?? 0);
        $paid   = (int) $this->payments()
            ->whereIn('status', [PaymentStatus::Succeeded->value, PaymentStatus::Captured->value])
            ->sum('amount_cents');

        $owing = max(0, $charge - $paid);

        return 'NZ$ ' . number_format($owing / 100, 2);
    }

    public function recalcTotalsFromPayments(): void
    {
        $paid = (int) $this->payments()
            ->whereIn('status', [PaymentStatus::Succeeded->value, PaymentStatus::Captured->value])
            ->sum('amount_cents');

        $total     = (int) ($this->charge_amount_cents ?? 0);
        $remaining = max(0, $total - $paid);

        if (Schema::hasColumn($this->getTable(), 'paid_amount_cents')) {
            $this->paid_amount_cents = $paid;
        }
        if (Schema::hasColumn($this->getTable(), 'remaining_amount_cents')) {
            $this->remaining_amount_cents = $remaining;
        }
        if (Schema::hasColumn($this->getTable(), 'balance_due_cents')) {
            $this->balance_due_cents = $remaining;
        }

        $this->saveQuietly();
    }

    /**
     * Fill money bucket fields from a VEVS booking payload.
     */
    public function fillMoneyFromVevs(array $vevs): void
    {
        $this->base_rental_cents       = (int) round(($vevs['car_rental_fee'] ?? 0) * 100);
        $this->extra_mileage_fee_cents = (int) round(($vevs['extra_mileage_fee'] ?? 0) * 100);
        $this->extras_cents            = (int) round(($vevs['extra_price'] ?? 0) * 100);

        $this->other_fees_cents        = (int) round(((
            $vevs['home_delivery_fee'] ?? 0
        ) + (
            $vevs['location_fee'] ?? 0
        ) + (
            $vevs['out_of_hours_fee'] ?? 0
        ) + (
            $vevs['late_return_fee'] ?? 0
        ) + (
            $vevs['young_driver_fee'] ?? 0
        ) + (
            $vevs['excess_price'] ?? 0
        )) * 100);

        $this->vat_cents         = (int) round(($vevs['vat_price'] ?? 0) * 100);
        $this->total_price_cents = (int) round(($vevs['total_price'] ?? 0) * 100);

        // Adjust keys if your payload is shaped differently
        $this->currency_code     = $vevs['currency']['value'] ?? $vevs['currency'] ?? 'NZD';

        // Extra extended bits while we're here (optional)
        $this->tax_cents                   = (int) round(($vevs['tax'] ?? 0) * 100);
        $this->total_before_discount_cents = (int) round(($vevs['total_price_before_discount'] ?? 0) * 100);
        $this->required_deposit_cents      = (int) round(($vevs['required_deposit'] ?? 0) * 100);
        $this->security_deposit_cents      = (int) round(($vevs['security_deposit'] ?? 0) * 100);
        $this->payment_method_external     = $vevs['payment_method'] ?? null;
        $this->external_status             = $vevs['status'] ?? null;
        $this->vat_rate                    = isset($vevs['vat']) ? (float) $vevs['vat'] : null;
    }

    public function chargeAmount(): Attribute
    {
        return Attribute::make(
            get: fn (): int => (int) ($this->charge_amount_cents ?? 0),
        );
    }

    public function holdAmount(): Attribute
    {
        return Attribute::make(
            get: fn (): int => (int) ($this->hold_amount_cents ?? 0),
        );
    }

    public function getHoldAmountCentsFromFlowAttribute(): int
    {
        return (int) (optional($this->flow)->hold_amount_cents ?? 0);
    }

    public function getDisplayRefAttribute(): string
    {
        return (string) ($this->external_reference ?: $this->reference ?: ('Job #' . $this->id));
    }

    public function getIsPaidInFullAttribute(): bool
    {
        $charge    = (int) ($this->charge_amount_cents ?? 0);
        $remaining = (int) $this->remaining_amount_cents;

        return $charge > 0 && $remaining === 0;
    }

    public function balanceDueCents(): Attribute
    {
        return Attribute::make(
            get: function (): int {
                // IMPORTANT: read the raw attribute to avoid recursion
                $raw = $this->getAttributeFromArray('balance_due_cents');
                if ($raw !== null) {
                    return (int) $raw;
                }

                $captured = (int) $this->payments()
                    ->whereIn('status', ['captured', 'succeeded'])
                    ->sum('amount_cents');

                return max(0, (int) ($this->charge_amount_cents ?? 0) - $captured);
            },
            set: fn ($value) => (int) $value,
        );
    }

    protected function remainingAmountCents(): Attribute
    {
        return Attribute::make(
            get: function ($value) {
                if (! is_null($value)) {
                    return (int) $value;
                }

                $charge = (int) ($this->charge_amount_cents ?? 0);
                $paid   = (int) $this->paid_amount_cents; // accessor above (cached-aware)

                return max($charge - $paid, 0);
            }
        );
    }

    /**
     * Resolve a human label for a given VEVS car_id.
     *
     * 1) Prefer a real Vehicle record (vehicles.vevs_car_id)
     * 2) Fall back to config/vevs_vehicles.php
     * 3) Fallback "VEVS car {id}"
     */
    public static function labelForVevsCarId(int|string|null $carId): ?string
    {
        if ($carId === null || $carId === '') {
            return null;
        }

        $carId = (string) $carId;

        // 1) Prefer a real Vehicle with matching vevs_car_id
        if ($vehicle = Vehicle::where('vevs_car_id', $carId)->first()) {
            // choose the nicest label available
            return $vehicle->display_name
                ?? $vehicle->plate
                ?? trim("{$vehicle->make} {$vehicle->model}")
                ?: "Vehicle {$carId}";
        }

        // 2) Fall back to config/vevs_vehicles.php
        $map = (array) config('vevs_vehicles', []);
        if (isset($map[$carId]) && $map[$carId] !== '') {
            return $map[$carId];
        }

        // 3) Sensible final fallback
        return 'VEVS car ' . $carId;
    }

    /**
     * Convenience accessor – what we want to show in the UI.
     *
     * Priority:
     *  1) Related Vehicle model (vehicle_id → vehicles table)
     *  2) vevs_car_id mapped via labelForVevsCarId()
     *  3) Raw `vehicle` column if it exists
     */
    public function getVehicleLabelAttribute(): ?string
    {
        // Try the loaded relationship first
        $vehicle = $this->getRelationValue('vehicle');

        // If it’s not loaded but we have an id, fetch it
        if (! $vehicle && $this->vehicle_id) {
            $vehicle = Vehicle::find($this->vehicle_id);
        }

        // 1) If we have a Vehicle model, build a clean label
        if ($vehicle) {
            $rego = $vehicle->registration; // e.g. "PJG211"
            $name = $vehicle->name;         // e.g. "Jimny 04"

            if ($rego && $name) {
                return "{$rego} – {$name}";
            }

            if ($rego) {
                return $rego;
            }

            if ($name) {
                return $name;
            }

            return "Vehicle {$vehicle->id}";
        }

        // 2) If no Vehicle, fall back to VEVS mapping
        if ($this->vevs_car_id) {
            return static::labelForVevsCarId($this->vevs_car_id);
        }

        // 3) Last resort: any legacy `vehicle` text column
        return $this->attributes['vehicle'] ?? null;
    }

    /* Scopes / Date helpers (continued) */

    public function startsWithinDays(int $days = 4): bool
    {
        if (! $this->start_at) {
            return false;
        }

        $start = $this->start_at instanceof Carbon
            ? $this->start_at
            : Carbon::parse((string) $this->start_at, config('app.timezone'));

        return now()->lte($start) && $start->lte(now()->addDays($days));
    }

    /* Deposit / Bond helpers */

    public function computePlannedCancelAt(?Carbon $baseEndAt = null): Carbon
    {
        $endAt = $baseEndAt
            ?? $this->end_at
            ?? optional($this->flow)->end_at
            ?? now();

        $minutes = app(AutoCancelHolds::class)->resolveFlowCancelMinutes(
            $this->flow,
            (int) config('holds.default_cancel_after_minutes', 720)
        );

        $planned = Carbon::parse($endAt, config('app.timezone'))->addMinutes($minutes);

        $authCap = now()->addDays((int) config('holds.stripe_auth_window_days', 7));
        if ($planned->gt($authCap)) {
            $planned = $authCap;
        }

        return $planned;
    }

    public function createAuthorizedDeposit(array $attributes = []): Deposit
    {
        $defaults = [
            'authorized_at'     => now(),
            'planned_cancel_at' => $this->computePlannedCancelAt(),
        ];

        if (! array_key_exists('job_id', $attributes)) {
            $attributes['job_id'] = $this->getKey();
        }

        return Deposit::create(array_replace($defaults, $attributes));
    }

    public function hasAuthorizedHold(): bool
    {
        return $this->deposits()
            ->whereNotNull('authorized_at')
            ->exists();
    }

    public function isBondHoldEligible(): bool
    {
        return $this->hold_amount_cents_from_flow > 0 && ! $this->hasAuthorizedHold();
    }

    public function payUrl(?int $amountCents = null): string
    {
        $params = ['job' => $this->getKey()];
        if ($amountCents !== null) {
            $params['amount_cents'] = $amountCents;
        }

        return URL::temporarySignedRoute(
            'portal.pay.show.job',
            now()->addDays(7),
            $params
        );
    }

    public function payHoldUrl(): string
    {
        $params = ['job' => $this->getKey(), 'bond_hold' => 1];

        return URL::temporarySignedRoute(
            'portal.pay.show.job',
            now()->addDays(7),
            $params
        );
    }

    /* External / VEVS helpers */

    /**
     * True if this Job originated from VEVS import.
     */
    public function isFromVevs(): bool
    {
        return $this->source_system === 'vevs';
    }

    /**
     * Convenience accessor for VEVS raw payload if you're stashing it there.
     */
    public function getVevsPayloadAttribute(): ?array
    {
        $payload = $this->external_payload ?? null;

        if (! is_array($payload)) {
            return null;
        }

        return $payload['vevs']['raw'] ?? null;
    }
}
