<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class OpsTask extends Model
{
    protected $fillable = [
        'title',
        'notes',
        'vehicle_id',
        'job_id',
        'brand_id',
        'for_user_id',
        'created_by_id',
        'completed_by_id',
        'due_date',
        'type',
        'priority',
        'completed_at',
    ];

    protected $casts = [
        'due_date'     => 'date',
        'completed_at' => 'datetime',
    ];

    /* Relationships */

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    public function forUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'for_user_id');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by_id');
    }

    public function completer(): BelongsTo
    {
        return $this->belongsTo(User::class, 'completed_by_id');
    }

    /* Scopes */

    public function scopeOpen($query)
    {
        return $query->whereNull('completed_at');
    }

    public function scopeCompletedToday($query)
    {
        return $query
            ->whereNotNull('completed_at')
            ->whereDate('completed_at', now()->toDateString());
    }

    public function scopeDueTodayOrOverdue($query)
    {
        return $query
            ->whereNull('completed_at')
            ->where(function ($q) {
                $q->whereNull('due_date')
                  ->orWhereDate('due_date', '<=', now()->toDateString());
            });
    }

    public function scopeOverdue($query)
    {
        return $query
            ->whereNull('completed_at')
            ->whereNotNull('due_date')
            ->whereDate('due_date', '<', now()->toDateString());
    }

    public function scopeForCurrentBrand($query)
    {
        // Safe placeholder: only applies if/when BrandContext exists
        if (class_exists(\App\Support\BrandContext::class)) {
            $brandId = \App\Support\BrandContext::currentBrandId();

            if ($brandId) {
                $query->where('brand_id', $brandId);
            }
        }

        return $query;
    }

    /* Helpers */

    public function getPriorityLabelAttribute(): string
    {
        return match ($this->priority) {
            1 => 'High',
            3 => 'Low',
            default => 'Normal',
        };
    }

    public function getPriorityColorAttribute(): string
    {
        // Tailwind-ish color names for badges
        return match ($this->priority) {
            1 => 'danger',
            3 => 'gray',
            default => 'warning',
        };
    }

    public function getTypeLabelAttribute(): string
    {
        return ucfirst($this->type ?: 'other');
    }

    public function isOverdue(): bool
    {
        return $this->due_date
            && $this->due_date->isPast()
            && ! $this->completed_at;
    }
}
