<?php

declare(strict_types=1);

namespace App\Models;

use App\Enums\OwnerLedgerEntryType;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;

class OwnerLedgerEntry extends Model
{
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'owner_id',
        'vehicle_id',
        'job_id',
        'payout_id',
        'type',
        'amount_cents',
        'owner_share_cents',   // ✅ NEW
        'currency_code',
        'description',
        'occurred_at',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'type'        => OwnerLedgerEntryType::class,
        'occurred_at' => 'datetime',
    ];

    /**
     * Back-link to the maintenance log (helps with clean-up).
     */
    public function maintenanceLog(): HasOne
    {
        return $this->hasOne(MaintenanceLog::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(Owner::class);
    }

    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    public function payout(): BelongsTo
    {
        return $this->belongsTo(OwnerPayout::class);
    }

    /**
     * Effective owner share for this entry.
     * - If owner_share_cents is set, use that.
     * - Otherwise, fall back to vehicle.owner_share_percent or 100%.
     */
    public function getEffectiveOwnerShareCentsAttribute(): int
    {
        // Explicit override wins (e.g. extra mileage = 100% owner)
        if ($this->owner_share_cents !== null) {
            return (int) $this->owner_share_cents;
        }

        $rawCents = (int) $this->amount_cents;
        $type     = $this->type instanceof OwnerLedgerEntryType
            ? $this->type->value
            : (string) $this->type;

        // Costs + payouts always 100% applied to owner
        if (in_array($type, ['cost', 'payout'], true)) {
            return $rawCents;
        }

        // Earnings – derive from vehicle owner_share_percent when set
        if ($type === 'earning') {
            $percent = $this->vehicle?->owner_share_percent;

            if ($percent !== null) {
                return (int) round($rawCents * ($percent / 100));
            }
        }

        // Fallback: full amount
        return $rawCents;
    }
}
