<?php
// app/Models/Payment.php

declare(strict_types=1);

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Schema;

/**
 * @property int|null    $id
 * @property int|null    $tenant_id
 * @property string|null $type
 * @property string|null $purpose
 * @property string      $currency
 * @property string      $status
 * @property int|null    $amount_cents
 * @property string|null $amount
 * @property array       $meta
 * @property array       $metadata
 * @property array       $details
 * @property \Carbon\CarbonImmutable|\Carbon\Carbon|null $paid_at
 */
class Payment extends Model
{
    use HasFactory;
    use SoftDeletes;

    /** Legacy heuristic threshold: if "amount" is an int above this, treat as cents */
    private const LEGACY_CENTS_THRESHOLD = 50_000;

    /** Cache schema lookups per-request to avoid repeated hasColumn calls */
    protected static array $schemaCache = [];

    /** @var array<int,string> */
    protected $guarded = [];

    /** Default attributes */
    protected $attributes = [
        'currency' => 'NZD',
        'status'   => 'pending',

        // JSON-backed attributes must be strings by default so Eloquent can cast them
        'metadata' => '[]',
        'meta'     => '[]',
        'details'  => '[]',
    ];

    /** Casts */
    protected $casts = [
        'metadata'    => 'array',
        'meta'        => 'array',
        'details'     => 'array',
        'paid_at'     => 'datetime',
        'one_off'     => 'boolean',
        'proof_paths' => 'array',
    ];

    /* -----------------------------------------------------------------
     |  Relationships
     | -----------------------------------------------------------------
     */

    public function link(): HasOne
    {
        return $this->hasOne(PaymentLink::class);
    }

    public function attachments(): MorphMany
    {
        return $this->morphMany(Attachment::class, 'attachable');
    }

    public function booking(): BelongsTo
    {
        return $this->belongsTo(Booking::class);
    }

    public function job(): BelongsTo
    {
        return $this->belongsTo(Job::class);
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    /**
     * Helper to create (or recreate) a PaymentLink with a valid tenant_id.
     * Always prefer the payment's tenant_id; fall back to current tenant() helper.
     */
    public function createLink(): PaymentLink
    {
        $tenantId = $this->tenant_id ?? (function_exists('tenant') ? tenant()?->id : null);

        if ($tenantId === null) {
            throw new \RuntimeException('Cannot create PaymentLink without a tenant_id');
        }

        // Replace any existing link (prevents duplicates/unique issues)
        if ($this->relationLoaded('link') ? ($this->link !== null) : $this->link()->exists()) {
            $this->link()->delete();
            $this->unsetRelation('link');
        }

        return $this->link()->create([
            'tenant_id'    => $tenantId,
            'public_token' => bin2hex(random_bytes(32)),
            'expires_at'   => now()->addDays(14),
        ]);
    }

    /* -----------------------------------------------------------------
     |  Column helpers
     | -----------------------------------------------------------------
     */

    protected function columnExists(string $column): bool
    {
        try {
            $table = $this->getTable();
            $key   = $table . ':' . $column;

            if (isset(self::$schemaCache[$key])) {
                return self::$schemaCache[$key];
            }

            $exists = Schema::hasTable($table) && Schema::hasColumn($table, $column);
            return self::$schemaCache[$key] = $exists;
        } catch (\Throwable) {
            return false;
        }
    }

    /* -----------------------------------------------------------------
     |  TYPE <-> PURPOSE (dual-field compatibility)
     | -----------------------------------------------------------------
     */

    public function getTypeAttribute()
    {
        if ($this->columnExists('type') && array_key_exists('type', $this->attributes)) {
            return $this->attributes['type'];
        }
        return $this->attributes['purpose'] ?? null;
    }

    public function setTypeAttribute($value): void
    {
        if ($this->columnExists('type')) {
            $this->attributes['type'] = $value;
        } else {
            $this->attributes['purpose'] = $value;
        }
    }

    public function getPurposeAttribute()
    {
        if ($this->columnExists('purpose') && array_key_exists('purpose', $this->attributes)) {
            return $this->attributes['purpose'];
        }
        return $this->attributes['type'] ?? null;
    }

    public function setPurposeAttribute($value): void
    {
        if ($this->columnExists('purpose')) {
            $this->attributes['purpose'] = $value;
        } else {
            $this->attributes['type'] = $value;
        }
    }

    /* -----------------------------------------------------------------
     |  AMOUNT_CENTS <-> AMOUNT (dual-field compatibility)
     | -----------------------------------------------------------------
     */

    public function getAmountCentsAttribute()
    {
        // Prefer native cents column if present
        if ($this->columnExists('amount_cents') && array_key_exists('amount_cents', $this->attributes)) {
            return $this->attributes['amount_cents'];
        }

        // Derive from legacy "amount"
        if (array_key_exists('amount', $this->attributes)) {
            $raw = $this->attributes['amount'];

            if ($raw === null) {
                return null;
            }

            // Legacy heuristic: suspiciously large "amount" is actually cents
            if (is_int($raw) && $raw > self::LEGACY_CENTS_THRESHOLD) {
                return $raw;
            }

            // Strings or floats treated as dollars (avoid float math where possible)
            if (is_string($raw)) {
                // normalize decimal string to cents using bcmath-like approach
                $normalized = number_format((float) $raw, 2, '.', '');
                return (int) round(((float) $normalized) * 100);
            }

            if (is_float($raw)) {
                return (int) round($raw * 100);
            }

            // Last resort cast
            return (int) round(((float) $raw) * 100);
        }

        return null;
    }

    public function setAmountCentsAttribute($value): void
    {
        $cents = match (true) {
            $value === null                                          => 0,
            is_int($value)                                           => $value,
            is_numeric($value) && str_contains((string) $value, '.') => (int) round(((float) $value) * 100),
            default                                                  => (int) $value,
        };

        if ($this->columnExists('amount_cents')) {
            $this->attributes['amount_cents'] = $cents;
        } else {
            $this->attributes['amount'] = number_format($cents / 100, 2, '.', '');
        }
    }

    /**
     * Return a decimal-string dollars value when no native amount column exists.
     * If a native decimal "amount" column exists, return that raw value.
     */
    public function getAmountAttribute()
    {
        if ($this->columnExists('amount') && array_key_exists('amount', $this->attributes)) {
            return $this->attributes['amount']; // expected to be a decimal string from DB
        }

        $cents = $this->getAmountCentsAttribute();
        return isset($cents) ? number_format($cents / 100, 2, '.', '') : null;
    }

    public function setAmountAttribute($value): void
    {
        $decimal = $value === null ? '0.00' : number_format((float) $value, 2, '.', '');

        if ($this->columnExists('amount')) {
            $this->attributes['amount'] = $decimal;
        } else {
            $this->attributes['amount_cents'] = (int) round(((float) $decimal) * 100);
        }
    }

    /* -----------------------------------------------------------------
     |  Normalizers
     | -----------------------------------------------------------------
     */

    public function setCurrencyAttribute($value): void
    {
        $this->attributes['currency'] = strtoupper((string) ($value ?: 'NZD'));
    }

    public function setStatusAttribute($value): void
    {
        $this->attributes['status'] = strtolower((string) $value);
    }

    /* -----------------------------------------------------------------
     |  Query scopes
     | -----------------------------------------------------------------
     */

    public function scopeSucceeded($q)
    {
        return $q->where('status', 'succeeded');
    }

    public function scopePending($q)
    {
        return $q->where('status', 'pending');
    }

    public function scopeFailed($q)
    {
        return $q->where('status', 'failed');
    }

    /* -----------------------------------------------------------------
     |  Model hooks
     | -----------------------------------------------------------------
     */

    protected static function booted(): void
    {
        static::saving(function (Payment $payment): void {
            // Ensure array casts are arrays (safeguard for early schema/data)
            if (! is_array($payment->metadata)) { $payment->metadata = (array) ($payment->metadata ?? []); }
            if (! is_array($payment->meta))     { $payment->meta     = (array) ($payment->meta ?? []); }
            if (! is_array($payment->details))  { $payment->details  = (array) ($payment->details ?? []); }

            // Auto-set paid_at when succeeded (idempotent)
            if (strtolower((string) $payment->status) === 'succeeded' && empty($payment->paid_at)) {
                $payment->paid_at = now();
            }
        });
    }

    /* -----------------------------------------------------------------
     |  Helpers
     | -----------------------------------------------------------------
     */

    public function getAmountFormattedAttribute(): string
    {
        $cur   = $this->currency ?: 'NZD';
        $cents = $this->getAmountCentsAttribute() ?? 0;

        return $cur . ' ' . number_format($cents / 100, 2, '.', ',');
    }

    public function isSucceeded(): bool { return ($this->status ?? '') === 'succeeded'; }
    public function isPending(): bool   { return ($this->status ?? '') === 'pending'; }
    public function isFailed(): bool    { return ($this->status ?? '') === 'failed'; }

    public function markSucceeded(): void
    {
        $this->status  = 'succeeded';
        $this->paid_at = $this->paid_at ?: now();
    }

    /* -----------------------------------------------------------------
     |  Domain actions
     | -----------------------------------------------------------------
     */

    /**
     * Mark a booking deposit as paid, schema-safely.
     * Optionally attach Stripe IDs if provided and columns exist.
     *
     * @param array{pi_id?:string,charge_id?:string} $ctx
     */
    public function markDepositAsPaid(array $ctx = []): void
    {
        if (($this->type ?? null) !== 'booking_deposit') {
            return;
        }

        // Attach Stripe identifiers if available and columns exist
        if ($this->columnExists('stripe_payment_intent_id')
            && !empty($ctx['pi_id'])
            && empty($this->stripe_payment_intent_id)) {
            $this->stripe_payment_intent_id = (string) $ctx['pi_id'];
        }

        if ($this->columnExists('stripe_charge_id')
            && !empty($ctx['charge_id'])
            && empty($this->stripe_charge_id)) {
            $this->stripe_charge_id = (string) $ctx['charge_id'];
        }

        // Normalize status + timestamps
        $this->status = 'succeeded';

        if ($this->columnExists('deposit_confirmed_at') && empty($this->deposit_confirmed_at)) {
            $this->deposit_confirmed_at = now();
        }

        if ($this->columnExists('captured_at') && empty($this->captured_at)) {
            $this->captured_at = now();
        }

        $this->save();
    }
}
