<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Vehicle extends Model
{
    protected $fillable = [
        // Core identity / display
        'name',
        'display_name',
        'plate',
        'registration',
        'vevs_vehicle_id',

        // Classification / usage
        'type',             // e.g. jimny_3door, jimny_5door, suv, sports, ev

        // Specs
        'make',
        'model',
        'year',
        'colour',
        'seats',
        'odometer_km',

        // Compliance / due fields
        'cof_due_at',
        'service_due_km',
        'rego_due_at',
        'service_due_at',

        // Registration / compliance extras
        'rego_type',
        'ruc_paid_to_km',
        'owner_share_percent',

        // Operational fields
        'lockbox_code',
        'gps_phone_number',
        'gps_topup_note',

        // Insurance / admin
        'insurance_provider',
        'appointments_note',

        // Flags / meta
        'is_active',
        'notes',

        // VEVS integration
        'vevs_car_id',

        // Ownership
        'owner_id',
        'ownership_type',
    ];

    protected $casts = [
        'cof_due_at'     => 'date',
        'rego_due_at'    => 'date',
        'service_due_at' => 'date',

        'is_active'          => 'bool',
        'owner_share_percent'=> 'float',
    ];

    protected static function booted(): void
    {
        static::saving(function (Vehicle $vehicle): void {
            // If the vehicle is not consignment, it should not be linked to an owner.
            if ($vehicle->ownership_type !== 'consignment') {
                $vehicle->owner_id = null;
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | Relationships
    |--------------------------------------------------------------------------
    */

    public function maintenanceLogs(): HasMany
    {
        return $this->hasMany(VehicleMaintenanceLog::class);
    }

    public function serviceLogs(): HasMany
    {
        return $this->maintenanceLogs()
            ->where('type', VehicleMaintenanceLog::TYPE_SERVICE);
    }

    public function cofLogs(): HasMany
    {
        return $this->maintenanceLogs()
            ->where('type', VehicleMaintenanceLog::TYPE_COF);
    }

    public function damages(): HasMany
    {
        return $this->hasMany(Damage::class);
    }

    public function jobs(): HasMany
    {
        return $this->hasMany(Job::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(Owner::class);
    }

    /**
     * Ledger entries charged to the owner for this specific vehicle.
     */
    public function ownerLedgerEntries(): HasMany
    {
        return $this->hasMany(OwnerLedgerEntry::class, 'vehicle_id');
    }

    /*
    |--------------------------------------------------------------------------
    | Helpers
    |--------------------------------------------------------------------------
    */

    public function isConsignment(): bool
    {
        return $this->ownership_type === 'consignment'
            && $this->owner_id !== null;
    }

    /**
     * Short label for tables / dropdowns.
     */
    public function getDisplayLabelAttribute(): string
    {
        // Prefer explicit display_name if set
        if ($this->display_name) {
            return $this->display_name;
        }

        // Registration + name combo (e.g. "QAA322  Jimny 01")
        $parts = [];

        if ($this->registration) {
            $parts[] = $this->registration;
        }

        if ($this->name) {
            $parts[] = ' ' . $this->name;
        }

        if ($parts) {
            return implode(' ', $parts);
        }

        // Fallback to make + model
        $basic = trim("{$this->make} {$this->model}");
        if ($basic !== '') {
            return $basic;
        }

        return "Vehicle {$this->id}";
    }

    /**
     * Verbose display name accessor.
     */
    public function getDisplayNameAttribute(): string
    {
        // If a custom display_name is set, use it.
        if (! empty($this->attributes['display_name'])) {
            return $this->attributes['display_name'];
        }

        $reg  = $this->registration ?: $this->plate;
        $name = $this->name ?: trim("{$this->make} {$this->model}");

        if ($reg && $name) {
            return "{$name} - {$reg}";
        }

        if ($reg) {
            return $reg;
        }

        if ($name) {
            return $name;
        }

        return "Vehicle {$this->id}";
    }
}
