<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class VehicleMaintenanceLog extends Model
{
    use HasFactory;

    protected $fillable = [
        'vehicle_id',
        'type',
        'title',
        'description',
        'odometer_km',
        'cost_cents',
        'performed_at',
        'next_due_at',
        'next_service_due_km',
        'attachment_path',

        // consignment / owner charging
        'charge_owner',
        'owner_ledger_entry_id',
    ];

    protected $casts = [
        'performed_at' => 'datetime',
        'next_due_at'  => 'datetime',
        'charge_owner' => 'bool',
    ];

    public const TYPE_SERVICE      = 'service';
    public const TYPE_COF          = 'cof';
    public const TYPE_REPAIR       = 'repair';
    public const TYPE_TYRES        = 'tyres';
    public const TYPE_OTHER        = 'other';
    public const TYPE_REGISTRATION = 'registration';
    public const TYPE_WINDSCREEN   = 'windscreen';

    /**
     * Options for the "Type" select.
     */
    public static function types(): array
    {
        return [
            self::TYPE_SERVICE      => 'Service',
            self::TYPE_COF          => 'COF / WOF',
            self::TYPE_REPAIR       => 'Repair',
            self::TYPE_TYRES        => 'Tyres',
            self::TYPE_WINDSCREEN   => 'Windscreen repair',
            self::TYPE_REGISTRATION => 'Registration',
            self::TYPE_OTHER        => 'Other',
        ];
    }

    /**
     * The vehicle this maintenance log belongs to.
     */
    public function vehicle(): BelongsTo
    {
        return $this->belongsTo(Vehicle::class);
    }

    /**
     * The owner ledger entry created when this maintenance
     * cost is charged back to a consignment owner.
     */
    public function ownerLedgerEntry(): BelongsTo
    {
        return $this->belongsTo(OwnerLedgerEntry::class);
    }

    /**
     * Convenience accessor: cost in dollars (string "123.45").
     */
    public function getCostDollarsAttribute(): ?string
    {
        if ($this->cost_cents === null) {
            return null;
        }

        return number_format($this->cost_cents / 100, 2);
    }

    // NOTE:
    // Vehicle odometer / current mileage / next_service_due_km
    // are now handled by VehicleMaintenanceLogObserver, not here.
}
