<?php

declare(strict_types=1);

namespace App\Observers;

use App\Models\VehicleMaintenanceLog;
use App\Services\OwnerMaintenanceChargeService;

class VehicleMaintenanceLogObserver
{
    public function __construct(
        protected OwnerMaintenanceChargeService $service,
    ) {}

    public function saved(VehicleMaintenanceLog $log): void
    {
        // 1. Existing behaviour: sync owner ledger
        $this->service->syncForLog($log);

        $vehicle = $log->vehicle;
        if (! $vehicle) {
            return;
        }

        $dirty = false;
        $type  = strtolower((string) $log->type);

        // 2. Keep current mileage in sync (use odometer_km column on vehicles)
        if (! is_null($log->odometer_km)) {
            $vehicle->odometer_km = max(
                (int) ($vehicle->odometer_km ?? 0),
                (int) $log->odometer_km
            );
            $dirty = true;
        }

        // 3. Service logs update next service kms (service_due_km on vehicles)
        if ($type === 'service' && ! is_null($log->next_service_due_km)) {
            $vehicle->service_due_km = (int) $log->next_service_due_km;
            $dirty = true;
        }

        // 4. COF logs update COF due date on the vehicle
        if ($type === 'cof' && ! is_null($log->next_due_at)) {
            $vehicle->cof_due_at = $log->next_due_at;   // <-- confirmed column
            $dirty = true;
        }

        // 5. Rego logs update rego due date on the vehicle
        if ($type === 'rego' && ! is_null($log->next_due_at)) {
            $vehicle->rego_due_at = $log->next_due_at;  // <-- adjust if your column is different
            $dirty = true;
        }

        if ($dirty) {
            $vehicle->save();
        }
    }

    public function deleted(VehicleMaintenanceLog $log): void
    {
        // 1. Existing behaviour: sync owner ledger cleanup
        $this->service->syncForLog($log);

        $vehicle = $log->vehicle;
        if (! $vehicle) {
            return;
        }

        // Recalculate from remaining logs

        // 2. service_due_km from latest remaining service log
        $latestService = $vehicle->maintenanceLogs()
            ->whereRaw('LOWER(type) = ?', ['service'])
            ->whereNotNull('next_service_due_km')
            ->latest('performed_at')
            ->first();

        $vehicle->service_due_km = $latestService?->next_service_due_km;

        // 3. COF due date from latest remaining COF log
        $latestCof = $vehicle->maintenanceLogs()
            ->whereRaw('LOWER(type) = ?', ['cof'])
            ->whereNotNull('next_due_at')
            ->latest('performed_at')
            ->first();

        $vehicle->cof_due_at = $latestCof?->next_due_at;

        // 4. Rego due date from latest remaining Rego log
        $latestRego = $vehicle->maintenanceLogs()
            ->whereRaw('LOWER(type) = ?', ['rego'])
            ->whereNotNull('next_due_at')
            ->latest('performed_at')
            ->first();

        $vehicle->rego_due_at = $latestRego?->next_due_at;

        $vehicle->save();
    }
}
