<?php

declare(strict_types=1);

namespace App\Services;

use App\Models\Job;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Throwable;

class ExternalSyncService
{
    public function __construct(
        protected VevsClientFactory $vevsFactory,
    ) {
    }

    /**
     * Import / refresh a single booking from VEVS by its reference.
     *
     * This is what your "Sync by Reference" button is calling.
     * Under the hood it just delegates to VeVsImporter::importByRef().
     *
     * @return array<string,mixed> The normalized row used for the upsert.
     *
     * @throws \Throwable
     */
    public function syncByReference(string $reference): array
    {
        try {
            /** @var VeVsImporter $importer */
            $importer = app(VeVsImporter::class);

            return $importer->importByRef($reference);
        } catch (Throwable $e) {
            Log::error('Failed syncing booking by reference from VEVS', [
                'reference' => $reference,
                'message'   => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Fetch the latest reservation from VEVS and update this Job,
     * including the VEVS pricing snapshot block.
     */
    public function syncJob(Job $job): void
    {
        try {
            $payload = $this->fetchVevsReservationForJob($job);

            if (! $payload) {
                return;
            }

            $this->updateJobFromVevsPayload($job, $payload);
        } catch (Throwable $e) {
            Log::error('Failed syncing Job from external system', [
                'job_id'  => $job->id,
                'message' => $e->getMessage(),
            ]);

            throw $e;
        }
    }

    /**
     * Resolve the correct VEVS client for the Job's brand and fetch its reservation.
     */
    protected function fetchVevsReservationForJob(Job $job): ?array
    {
        // Adjust these to however you actually store brand identifiers:
        $brandKey = $job->brand?->key ?? $job->brand_key ?? null;

        // VevsClientFactory decides which VEVS connection/config to use.
        $client = $this->vevsFactory->forBrandKey($brandKey);

        // Adjust this field if your Job uses a different name for the VEVS ref.
        return $client->fetchReservationByRef($job->booking_reference);
    }

    /**
     * Map VEVS payload → Job fields.
     * Adjust the attribute names on the left to match your jobs table.
     */
    protected function updateJobFromVevsPayload(Job $job, array $data): void
    {
        $job->fill([
            // Currency
            'vevs_currency'                    => Arr::get($data, 'currency.value') ?? $job->vevs_currency,

            // Core rental pieces
            'vevs_base_rental_cents'           => $this->toCents(Arr::get($data, 'car_rental_fee')),
            'vevs_extras_cents'                => $this->toCents(Arr::get($data, 'extra_price')),
            'vevs_tax_cents'                   => $this->toCents(Arr::get($data, 'tax')),
            'vevs_vat_rate_percent'            => $this->toFloat(Arr::get($data, 'vat')),
            'vevs_vat_amount_cents'            => $this->toCents(Arr::get($data, 'vat_price')),
            'vevs_total_before_discount_cents' => $this->toCents(Arr::get($data, 'total_price_before_discount')),
            'vevs_total_cents'                 => $this->toCents(Arr::get($data, 'total_price')),

            // Deposits
            'vevs_required_deposit_cents'      => $this->toCents(Arr::get($data, 'required_deposit')),
            'vevs_security_deposit_cents'      => $this->toCents(Arr::get($data, 'security_deposit')),

            // Metadata
            'vevs_payment_method'              => Arr::get($data, 'payment_method'),
            'vevs_status'                      => Arr::get($data, 'status'),
        ]);

        $job->save();
    }

    protected function toCents(mixed $value): ?int
    {
        if ($value === null || $value === '') {
            return null;
        }

        // VEVS gives us strings like "123.45"
        $float = (float) $value;

        return (int) round($float * 100);
    }

    protected function toFloat(mixed $value): ?float
    {
        if ($value === null || $value === '') {
            return null;
        }

        return (float) $value;
    }
}
