<?php

declare(strict_types=1);

namespace App\Services;

use App\Models\OwnerLedgerEntry;
use App\Models\VehicleMaintenanceLog;
use Illuminate\Support\Facades\DB;

class OwnerMaintenanceChargeService
{
    /**
     * Ensure the owner ledger is in sync with this maintenance log.
     *
     * - For consignment vehicles with an owner, cost, and charge_owner=true:
     *     → create/update an OwnerLedgerEntry of type "cost".
     * - Otherwise:
     *     → delete any existing linked OwnerLedgerEntry.
     */
    public function syncForLog(VehicleMaintenanceLog $log): void
    {
        $log->loadMissing('vehicle.owner', 'ownerLedgerEntry');

        $vehicle = $log->vehicle;

        // If the log is no longer attached to a vehicle, remove any ledger entry.
        if (! $vehicle) {
            $this->detachExistingLedgerEntry($log);

            return;
        }

        // Only consignment vehicles with an owner and a positive cost are eligible.
        $isConsignment = $vehicle->ownership_type === 'consignment';
        $hasOwner      = (bool) $vehicle->owner_id;
        $hasCost       = (int) $log->cost_cents > 0;

        // If we're not meant to charge the owner OR it's not a valid consignment scenario,
        // remove any existing ledger entry and bail.
        if (! $log->charge_owner || ! $isConsignment || ! $hasOwner || ! $hasCost) {
            $this->detachExistingLedgerEntry($log);

            return;
        }

        DB::transaction(function () use ($log, $vehicle) {
            /** @var \App\Models\OwnerLedgerEntry|null $entry */
            $entry = $log->ownerLedgerEntry;

            $payload = [
                'owner_id'     => $vehicle->owner_id,
                'vehicle_id'   => $vehicle->id,
                'type'         => 'cost',
                'amount_cents' => (int) $log->cost_cents,
                'currency'     => 'NZD',
                'description'  => 'Maintenance: ' . ($log->title ?: 'Unknown'),
                'occurred_at'  => $log->performed_at ?? now(),
            ];

            if ($entry) {
                $entry->fill($payload)->save();
            } else {
                $entry = OwnerLedgerEntry::create($payload);

                // Avoid recursion by updating quietly.
                $log->owner_ledger_entry_id = $entry->id;
                $log->saveQuietly();
            }
        });
    }

    /**
     * Remove any linked OwnerLedgerEntry when we no longer want to charge the owner.
     */
    protected function detachExistingLedgerEntry(VehicleMaintenanceLog $log): void
    {
        if (! $log->owner_ledger_entry_id) {
            return;
        }

        $entryId = $log->owner_ledger_entry_id;

        $log->owner_ledger_entry_id = null;
        $log->saveQuietly();

        OwnerLedgerEntry::whereKey($entryId)->delete();
    }
}
