<?php

declare(strict_types=1);

namespace App\Services\Payments;

use App\Models\Job;
use App\Models\Payment;
use App\Models\PaymentLink;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;

// If the Stripe SDK is installed, we'll use the client class.
// composer require stripe/stripe-php
use Stripe\Stripe;
use Stripe\StripeClient;
use Stripe\Exception\ApiErrorException;

class DepositProvisioner
{
    /**
     * Ensure a deposit Payment exists for the job if the flow requires a hold.
     * Idempotent: returns the existing deposit if found.
     *
     * @param  Job   $job
     * @param  array $ctx  Options:
     *   - provider: 'stripe' | ...
     *   - planned_authorize_at, planned_release_at (stored in details if column exists)
     *   - authorize_now: bool (default true) create a manual-capture PI immediately for stripe
     */
    public function ensureForJob(Job $job, array $ctx = []): ?Payment
    {
        $holdCents = $this->resolveHoldCents($job);
        if ($holdCents <= 0) {
            return null; // no hold configured
        }

        // Don’t duplicate: return most-recent non-deleted deposit for this job
        $existing = Payment::query()
            ->where('job_id', $job->id)
            ->where('type', 'booking_deposit')
            ->whereNull('deleted_at')
            ->latest('id')
            ->first();

        if ($existing) {
            // Optionally (re)authorize now if caller requests and we don't have a PI yet
            if (
                ($ctx['authorize_now'] ?? true) &&
                ($existing->provider ?? 'stripe') === 'stripe' &&
                $this->needsStripeAuthorization($existing)
            ) {
                $this->createStripeHoldIntent($existing);
            }
            return $existing;
        }

        // Create deposit, only touching columns that exist in this schema
        $p = new Payment();

        if (Schema::hasColumn('payments', 'tenant_id')) {
            $p->tenant_id = $job->tenant_id ?? (function_exists('tenant') && tenant() ? tenant()->id : null);
        }

        $p->job_id       = $job->id;
        $p->type         = 'booking_deposit';
        $p->status       = 'pending';
        $p->mechanism    = 'card';
        $p->provider     = $ctx['provider'] ?? config('payments.default', 'stripe');
        $p->currency     = $job->currency ?? 'NZD';
        $p->amount_cents = $holdCents;
        $p->reference    = $job->external_reference ?? null;

        if (Schema::hasColumn('payments', 'details')) {
            $p->details = array_filter([
                'source'               => 'provisioned_after_final_payment',
                'planned_authorize_at' => $ctx['planned_authorize_at'] ?? null,
                'planned_release_at'   => $ctx['planned_release_at'] ?? null,
            ]);
        }

        $p->save();

        // Create a public link if table exists (and add tenant_id only if present)
        if (Schema::hasTable('payment_links')) {
            $attrs  = ['payment_id' => $p->id];
            $values = ['public_token' => bin2hex(random_bytes(32))];

            if (Schema::hasColumn('payment_links', 'expires_at')) {
                $values['expires_at'] = now()->addDays(30);
            }
            if (Schema::hasColumn('payment_links', 'tenant_id') && isset($p->tenant_id)) {
                $values['tenant_id'] = $p->tenant_id;
            }

            PaymentLink::firstOrCreate($attrs, $values);
        }

        // --- AUTHORIZE THE HOLD (Stripe manual-capture PI) -------------------
        if (($ctx['authorize_now'] ?? true) && ($p->provider === 'stripe')) {
            $this->createStripeHoldIntent($p);
        }

        return $p;
    }

    /**
     * Create a manual-capture Stripe PaymentIntent for the deposit and
     * persist identifiers + status back onto the Payment row.
     *
     * Mirrors the guidance:
     *  - capture_method: 'manual'
     *  - metadata: kind=deposit_auth, deposit_id, job_id, job_ref
     *  - save $pi->id to payments.stripe_payment_intent_id (if column exists)
     *  - set status to 'requires_capture' and capture_method = 'manual' (if column exists)
     */
    private function createStripeHoldIntent(Payment $deposit): void
    {
        // Already linked to a PI? Nothing to do.
        if (Schema::hasColumn('payments', 'stripe_payment_intent_id') && !empty($deposit->stripe_payment_intent_id)) {
            return;
        }

        // Configure Stripe
        $secret = (string) (config('services.stripe.secret')
            ?: config('payments.drivers.stripe.secret', ''));
        if ($secret === '') {
            Log::warning('Stripe secret not configured; skipping hold PI creation for deposit '.$deposit->id);
            return;
        }

        Stripe::setApiKey($secret);
        $client = new StripeClient($secret);

        $amount   = (int) $deposit->amount_cents;
        $currency = strtolower((string) ($deposit->currency ?? 'nzd'));

        try {
            $pi = $client->paymentIntents->create([
                'amount'         => $amount,
                'currency'       => $currency,
                'capture_method' => 'manual',
                // If you have a Customer or PaymentMethod to attach at auth time, add:
                // 'customer'       => $deposit->customer_id ?? null,
                // 'payment_method' => $pmId,
                // 'confirm'        => true, // only if you are confirming right now
                'metadata'       => [
                    'kind'       => 'deposit_auth',
                    'deposit_id' => (string) $deposit->id,
                    'job_id'     => (string) $deposit->job_id,
                    'job_ref'    => (string) ($deposit->reference ?? ''),
                ],
            ]);

            // persist identifiers + status on the deposit
            if (Schema::hasColumn('payments', 'stripe_payment_intent_id')) {
                $deposit->stripe_payment_intent_id = $pi->id;
            }

            // Mark as authorized/awaiting capture
            $deposit->status = 'requires_capture';
            if (Schema::hasColumn('payments', 'capture_method')) {
                $deposit->capture_method = 'manual';
            }

            $deposit->save();
        } catch (ApiErrorException $e) {
            Log::error('Stripe PI create failed for deposit '.$deposit->id.': '.$e->getMessage(), [
                'deposit_id' => $deposit->id,
                'job_id'     => $deposit->job_id,
                'code'       => $e->getError() ? $e->getError()->code : null,
                'type'       => $e->getError() ? $e->getError()->type : null,
            ]);
            // Leave the deposit in 'pending' so a retry path can pick it up later.
        }
    }

    /** Determine whether we still need to create a Stripe manual-capture PI. */
    private function needsStripeAuthorization(Payment $deposit): bool
    {
        if ($deposit->provider !== 'stripe') {
            return false;
        }
        if (Schema::hasColumn('payments', 'stripe_payment_intent_id') && !empty($deposit->stripe_payment_intent_id)) {
            return false;
        }
        // Only attempt if we have a sensible amount
        return (int) $deposit->amount_cents > 0;
    }

    /** Find the hold amount (job override, flow columns, or flow JSON). */
    private function resolveHoldCents(Job $job): int
    {
        foreach (['hold_amount_cents', 'hold_cents', 'security_hold_cents'] as $k) {
            if (isset($job->{$k}) && (int) $job->{$k} > 0) {
                return (int) $job->{$k};
            }
        }

        $flow = $job->flow ?? null;
        if ($flow) {
            foreach (['hold_amount_cents', 'hold_cents', 'security_hold_cents'] as $k) {
                if (isset($flow->{$k}) && (int) $flow->{$k} > 0) {
                    return (int) $flow->{$k};
                }
            }
            foreach (['settings', 'config', 'options', 'meta'] as $json) {
                $v   = $flow->{$json} ?? null;
                $arr = is_array($v) ? $v : (is_string($v) ? json_decode($v, true) : null);
                if (is_array($arr)) {
                    foreach (['hold_amount_cents', 'hold_cents', 'security_hold_cents', 'hold_amount'] as $k) {
                        if (isset($arr[$k]) && (int) $arr[$k] > 0) {
                            return (int) $arr[$k];
                        }
                    }
                }
            }
        }

        return 0;
    }
}
