<?php

namespace App\Services\Payments\Gateways;

use App\Models\Payment;
use App\Services\Payments\PaymentGateway;
use App\Services\Payments\SessionResult;
use Illuminate\Support\Str;

/**
 * Minimal PxPay 2.0 HPP create/redirect flow.
 * Docs: https://www.windcave.com/developer-e-commerce-pxpay2
 */
class WindcaveGateway implements PaymentGateway
{
    public function __construct(private array $cfg) {}

    private function post(string $path, array $payload): array
    {
        $url = "https://sec.windcave.com/api/v2/pxpay{$path}";
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'Authorization: Basic '.base64_encode($this->cfg['user_id'].':'.$this->cfg['key']),
            ],
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload),
            CURLOPT_TIMEOUT => 20,
        ]);
        $raw = curl_exec($ch);
        if ($raw === false) throw new \RuntimeException('Windcave request failed: '.curl_error($ch));
        $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        curl_close($ch);
        $data = json_decode($raw, true) ?: [];
        if ($code >= 300) throw new \RuntimeException('Windcave API error: '.$code.' '.($data['responseText'] ?? $raw));
        return $data;
    }

    public function createOneOffSession(Payment $p): SessionResult
    {
        $amount = (int) ($p->amount_cents ?? 0);
        if ($amount <= 0) throw new \RuntimeException('Amount must be > 0');

        $merchantRef = Str::limit($p->reference ?? ('PAY-'.$p->id), 64, '');

        $req = [
            'amount'      => number_format($amount/100, 2, '.', ''), // string "12.34"
            'currency'    => strtoupper($p->currency ?? 'NZD'),
            'merchantReference' => $merchantRef,
            'txnType'     => 'Purchase',
            'transactionId' => (string) $p->id, // optional; helps reconcile
            'emailAddress' => $p->meta['customer_email'] ?? null,
            'urls' => [
                'success' => $this->cfg['return_url'].'?status=success&pid='.$p->id,
                'fail'    => $this->cfg['return_url'].'?status=fail&pid='.$p->id,
                'notify'  => $this->cfg['notify_url'], // optional server-to-server
            ],
        ];

        $res = $this->post('/create', array_filter($req));

        // Windcave returns 'uri' (redirect URL) and 'validity'
        $p->provider    = 'windcave';
        $p->provider_id = $res['transactionId'] ?? null; // keep their id if returned
        $p->status = $p->status ?: 'pending';
        $p->save();

        return new SessionResult('windcave', redirectUrl: $res['uri'] ?? null, extra: $res);
    }

    public function authorizeHold(Payment $p): SessionResult
    {
        // Similar to Purchase, but TxnType='Auth'
        $amount = (int) ($p->amount_cents ?? 0);
        $req = [
            'amount'      => number_format($amount/100, 2, '.', ''),
            'currency'    => strtoupper($p->currency ?? 'NZD'),
            'merchantReference' => $p->reference ?? ('HOLD-'.$p->id),
            'txnType'     => 'Auth',
            'transactionId' => (string) $p->id,
            'urls' => [
                'success' => $this->cfg['return_url'].'?status=success&pid='.$p->id.'&kind=auth',
                'fail'    => $this->cfg['return_url'].'?status=fail&pid='.$p->id.'&kind=auth',
                'notify'  => $this->cfg['notify_url'],
            ],
        ];
        $res = $this->post('/create', $req);
        $p->provider = 'windcave';
        $p->provider_id = $res['transactionId'] ?? null;
        $p->status = 'pending';
        $p->save();

        return new SessionResult('windcave', redirectUrl: $res['uri'] ?? null, extra: $res);
    }

    public function captureHold(Payment $p, ?int $amountCents = null): void
    {
        // PxPay capture API (Complete) – depends on merchant setup; often done via PxPost
        // Provide a no-op or implement PxPost/Capture endpoint as required.
        throw new \RuntimeException('Windcave captureHold not implemented in demo.');
    }

    public function cancelHold(Payment $p): void
    {
        throw new \RuntimeException('Windcave cancelHold not implemented in demo.');
    }

    public function refund(Payment $p, ?int $amountCents = null): void
    {
        throw new \RuntimeException('Windcave refund not implemented in demo.');
    }
}
