<?php 

namespace App\Services\Payments;

use App\Models\Payment;
use Stripe\StripeClient;
use Illuminate\Support\Arr;

class StripeHoldService
{
    public function __construct(private StripeClient $stripe) {}

    /**
     * Authorize (not capture) the deposit hold for a Payment row of type "booking_deposit".
     * $pmId can be a saved PM or you can omit it if you use AME.
     */
    public function authorizeHold(Payment $deposit, array $opts = []): Payment
    {
        // Safety rails
        if ($deposit->type !== 'booking_deposit') {
            throw new \InvalidArgumentException('Payment is not a booking_deposit');
        }
        if ($deposit->amount_cents <= 0) {
            throw new \RuntimeException('Deposit amount must be > 0');
        }

        $currency = strtolower($deposit->currency ?: 'nzd');
        $pmId     = Arr::get($opts, 'payment_method_id');   // optional if using AME
        $customer = Arr::get($opts, 'customer_id');         // optional

        // Build metadata so webhooks can find the record quickly
        $metadata = [
            'kind'       => 'deposit_auth',
            'deposit_id' => (string) $deposit->id,
            'job_id'     => (string) $deposit->job_id,
            'job_ref'    => (string) ($deposit->reference ?? ''),
            'tenant_id'  => (string) ($deposit->tenant_id ?? ''),
        ];

        // Create a manual-capture PI for the hold
        $params = [
            'amount'               => $deposit->amount_cents,
            'currency'             => $currency,
            'capture_method'       => 'manual',
            'confirm'              => true,
            'metadata'             => $metadata,
            'automatic_payment_methods' => ['enabled' => true], // or pass payment_method types explicitly
        ];
        if ($pmId)    { $params['payment_method'] = $pmId; }
        if ($customer){ $params['customer']       = $customer; }

        $pi = $this->stripe->paymentIntents->create($params);

        // ⬇⬇ THIS is the bit you asked about — save the Stripe IDs on the deposit
        $deposit->stripe_payment_intent_id = $pi->id;
        $deposit->stripe_payment_method_id = $pmId ?: ($pi->payment_method ?? null);
        $deposit->capture_method           = 'manual';
        $deposit->status                   = 'requires_capture'; // or 'authorized'
        $deposit->authorized_at            = now();
        $deposit->save();

        return $deposit;
    }
}
