<?php
// app/Services/Payments/StripeService.php

declare(strict_types=1);

namespace App\Services\Payments;

use App\Models\Payment;
use Illuminate\Support\Facades\Log;
use Stripe\Exception\ApiErrorException;
use Stripe\StripeClient;

/**
 * Stripe integration boundary for the Holds app.
 *
 * Responsibilities:
 *  - Create/confirm off-session PaymentIntents for one-off charges
 *  - Keep metadata/idempotency consistent
 *  - Update local Payment model on success/failure
 */
class StripeService
{
    public function __construct(
        private readonly StripeClient $stripe,
    ) {
        // Bind in a service provider:
        // $this->app->singleton(StripeClient::class, fn () => new StripeClient(config('services.stripe.secret')));
    }

    /**
     * Attempt a one-off, *off-session* charge against the Job's saved Stripe customer.
     * Throws on configuration/validation issues; returns the updated Payment on Stripe success.
     *
     * Outcomes:
     *  - Succeeds: marks Payment as 'succeeded', sets paid_at, stores payment_intent id
     *  - Fails with card error: throws; caller can surface a friendly message
     *  - Requires action: off-session should *not* require action — treat as failure and fall back to request flow
     */
    public function attemptOffSessionCharge(Payment $payment): Payment
    {
        $job = $payment->job;
        $customerStripeId = optional($job?->customer)->stripe_customer_id;

        abort_unless($customerStripeId, 422, 'No saved card on file. Send a payment request instead.');

        $amount = (int) $payment->amount_cents;        // integer cents
        $currency = strtolower($payment->currency ?? 'nzd');

        $metadata = $this->buildMetadata($payment);

        // Make idempotency stable per Payment attempt
        $idem = sprintf('pmt:%d:%s', $payment->id, sha1('oneoff|'.$amount.'|'.$currency));

        try {
            $pi = $this->stripe->paymentIntents->create([
                'amount'               => $amount,
                'currency'             => $currency,
                'customer'             => $customerStripeId,
                'off_session'          => true,
                'confirm'              => true,
                'confirmation_method'  => 'automatic',
                'metadata'             => $metadata,
                // If you want to constrain to a specific default payment method:
                // 'payment_method'     => $job->customer->default_payment_method_id,
                // 'payment_method_types' => ['card'],
            ], ['idempotency_key' => $idem]);

            // Save identifiers regardless of outcome
            $payment->stripe_payment_intent_id = $pi->id;
            $payment->saveQuietly();

            if ($pi->status === 'succeeded' || $pi->status === 'requires_capture') {
                // For automatic confirmation + off_session, you'll usually get 'succeeded'
                $payment->forceFill([
                    'status'  => 'succeeded',
                    'paid_at' => now(),
                ])->save();

                return $payment;
            }

            // Any other status off-session is effectively a failure for our flow
            Log::warning('[Stripe] Off-session charge did not succeed', [
                'payment_id' => $payment->id,
                'pi_status'  => $pi->status,
                'pi_id'      => $pi->id,
            ]);

            abort(402, 'Card could not be charged off-session. Send a payment request instead.');
        } catch (ApiErrorException $e) {
            // Capture useful context but avoid leaking secrets
            Log::error('[Stripe] Off-session charge error', [
                'payment_id' => $payment->id,
                'stripe_code'=> $e->getStripeCode(),
                'type'       => $e->getError()?->type,
                'decline_code' => $e->getError()?->decline_code,
                'message'    => $e->getMessage(),
            ]);

            // Common off-session card errors (authentication_required, card_declined, insufficient_funds...)
            abort(402, $this->friendlyStripeMessage($e));
        }
    }

    /**
     * Minimal metadata to tie Stripe objects back to our system.
     */
    private function buildMetadata(Payment $payment): array
    {
        $job = $payment->job;

        return array_filter([
            'kind'       => 'one_off',
            'payment_id' => (string) $payment->id,
            'job_id'     => (string) ($job->id ?? ''),
            'job_ref'    => (string) ($job->ref ?? ''),
            'brand'      => (string) ($job->brand ?? ''),
        ], fn ($v) => $v !== '');
    }

    /**
     * Map Stripe exceptions to short, user-facing messages.
     */
    private function friendlyStripeMessage(ApiErrorException $e): string
    {
        $code = $e->getStripeCode() ?: $e->getError()?->code;

        return match ($code) {
            'authentication_required' => 'Your bank needs you to approve this payment. Please open the payment request link instead.',
            'card_declined'           => 'The card was declined. Please try a different card via the payment request link.',
            'insufficient_funds'      => 'There are not enough funds on the card. Please try again with another card.',
            'generic_decline'         => 'The card was declined. Please try another card.',
            default                   => 'We couldn’t charge the saved card. Please use the payment request link.',
        };
    }
}
