<?php

namespace App\Services\Pricing;

use App\Models\PricingDecision;
use App\Models\Vehicle;
use App\Support\Pricing\PricingQuote;
use Carbon\Carbon;

class PricingEngine
{
    /**
     * Main entry point.
     *
     * - Called by booking portals (Jimny + Dream Drives).
     * - Works with both "category" pricing (Jimny 3/5 door) and per-vehicle pricing.
     * - Logs every decision into pricing_decisions for future AI training.
     */
    public function quoteDailyRate(
        string $brand,
        ?Vehicle $vehicle,
        ?string $vehicleType,
        string $pickupAt,
        string $returnAt,
        array $context = [],
    ): PricingQuote {
        $pickup = Carbon::parse($pickupAt);
        $return = Carbon::parse($returnAt);

        // Minimum 1 day charge.
        $days = max(1, $pickup->diffInDays($return));

        // 1) Build a feature set (what the AI will eventually see).
        $features = $this->buildFeatureSet(
            brand: $brand,
            vehicle: $vehicle,
            vehicleType: $vehicleType,
            pickup: $pickup,
            return: $return,
            context: $context,
        );

        // 2) Base daily rate (config / per-vehicle).
        $baseDailyRate = $this->baseDailyRate(
            brand: $brand,
            vehicle: $vehicle,
            vehicleType: $vehicleType,
            features: $features,
        );

        // 3) Apply adjustment pipeline (lead time, seasonality, events, demand, competitors, AI).
        $meta = [
            'brand'        => $brand,
            'vehicle_id'   => $vehicle?->id,
            'vehicle_type' => $vehicleType,
            'days'         => $days,
            'rules_version'=> 'v1-skeleton', // bump when you change logic
            'base_daily'   => $baseDailyRate,
        ];

        $adjustedDailyRate = $this->applyAdjustments(
            baseDailyRate: $baseDailyRate,
            features: $features,
            meta: $meta,
        );

        // 4) Clamp to guardrails (safety).
        $finalDailyRate = $this->applyGuards(
            dailyRate: $adjustedDailyRate,
            brand: $brand,
            features: $features,
            meta: $meta,
        );

        $total = $days * $finalDailyRate;

        // 5) Log decision for training / audit.
        PricingDecision::create([
            'brand'            => $brand,
            'vehicle_id'       => $vehicle?->id,
            'vehicle_type'     => $vehicleType,
            'pickup_at'        => $pickup,
            'return_at'        => $return,
            'daily_rate_cents' => $finalDailyRate,
            'total_cents'      => $total,
            'source'           => $context['source'] ?? null,
            'context'          => $meta + ['features' => $features],
        ]);

        // 6) Return a structured quote.
        return new PricingQuote(
            dailyRateCents: $finalDailyRate,
            totalCents: $total,
            meta: $meta,
        );
    }

    /**
     * Build the "feature vector" – everything you'd eventually feed to an AI model.
     * For now it's mostly metadata, but we structure it properly from day one.
     */
    protected function buildFeatureSet(
        string $brand,
        ?Vehicle $vehicle,
        ?string $vehicleType,
        Carbon $pickup,
        Carbon $return,
        array $context = [],
    ): array {
        $days = max(1, $pickup->diffInDays($return));
        $leadTimeDays = max(0, now()->diffInDays($pickup, false));

        return [
            'brand'          => $brand,
            'vehicle_id'     => $vehicle?->id,
            'vehicle_type'   => $vehicleType ?? $vehicle?->type,
            'vehicle_year'   => $vehicle?->year,
            'vehicle_km'     => $vehicle?->odometer_km,
            'vehicle_base'   => $vehicle?->daily_rate_cents, // for Dream Drives

            'pickup_at'      => $pickup->toAtomString(),
            'return_at'      => $return->toAtomString(),
            'days'           => $days,
            'lead_time_days' => $leadTimeDays,

            // Calendar features
            'pickup_day_of_week' => $pickup->dayOfWeekIso, // 1..7
            'pickup_month'       => $pickup->month,
            'is_weekend'         => (int) $pickup->isWeekend(),

            // Placeholders for future signals
            'is_school_holiday'  => false,
            'is_public_holiday'  => false,
            'local_events_score' => 0.0,

            'demand_score'       => 0.0,   // utilisation
            'competitor_score'   => 0.0,   // relative to competitors

            // Any extra context from caller (channel, promo code, etc.)
            'context'            => $context,
        ];
    }

    /**
     * Base daily rate before any multipliers.
     *
     * - Jimny: from config by type (3-door / 5-door).
     * - Dream Drives: from vehicle.daily_rate_cents, fallback to config default.
     * - Other brands: generic default.
     */
    protected function baseDailyRate(
        string $brand,
        ?Vehicle $vehicle,
        ?string $vehicleType,
        array $features,
    ): int {
        // Jimny: type-based
        if ($brand === 'jimny') {
            $config = config('pricing.jimny', []);

            $typeKey = $vehicleType
                ?? $vehicle?->type
                ?? 'jimny_3door';

            if (isset($config[$typeKey]) && is_int($config[$typeKey])) {
                return (int) $config[$typeKey];
            }

            // Fallback Jimny rate
            return (int) ($config['jimny_3door'] ?? 16000);
        }

        // Dream Drives: per-vehicle, fall back to brand default.
        if ($brand === 'dreamdrives') {
            if ($vehicle && $vehicle->daily_rate_cents) {
                return (int) $vehicle->daily_rate_cents;
            }

            $ddDefault = config('pricing.dreamdrives.default');
            if (is_int($ddDefault)) {
                return (int) $ddDefault;
            }

            return 30000;
        }

        // Other brands: generic fallback.
        return 30000;
    }

    /**
     * Apply the full adjustment pipeline to the base rate.
     *
     * Currently each adjustment is a stub returning a 1.0 multiplier,
     * but the structure is ready for real logic / AI.
     */
    protected function applyAdjustments(
        int $baseDailyRate,
        array $features,
        array &$meta,
    ): int {
        $rate = $baseDailyRate;
        $multipliers = [];

        // 1) Lead time (how far in advance they’re booking).
        $leadAdj = $this->leadTimeAdjustment($features);
        $rate = (int) round($rate * $leadAdj['multiplier']);
        $multipliers['lead_time'] = $leadAdj;

        // 2) Seasonality (month, summer / winter, ski, etc.).
        $seasonAdj = $this->seasonalityAdjustment($features);
        $rate = (int) round($rate * $seasonAdj['multiplier']);
        $multipliers['seasonality'] = $seasonAdj;

        // 3) Demand / utilisation (how busy you are).
        $demandAdj = $this->demandAdjustment($features);
        $rate = (int) round($rate * $demandAdj['multiplier']);
        $multipliers['demand'] = $demandAdj;

        // 4) Local events (concerts, festivals, etc.).
        $eventAdj = $this->eventAdjustment($features);
        $rate = (int) round($rate * $eventAdj['multiplier']);
        $multipliers['events'] = $eventAdj;

        // 5) Competitor pricing.
        $compAdj = $this->competitorAdjustment($features);
        $rate = (int) round($rate * $compAdj['multiplier']);
        $multipliers['competitors'] = $compAdj;

        // 6) AI / ML model hook – currently a no-op.
        $aiAdj = $this->aiModelAdjustment($features, $rate);
        $rate = (int) round($rate * $aiAdj['multiplier']);
        $multipliers['ai_model'] = $aiAdj;

        $meta['multipliers'] = $multipliers;

        return max($rate, 0);
    }

    /**
     * Guardrail clamping – makes sure prices never go insane.
     *
     * Values can be configured in config/pricing.php:
     *
     * 'guards' => [
     *     'min_daily_cents' => 10000,
     *     'max_daily_cents' => 100000,
     * ]
     */
    protected function applyGuards(
        int $dailyRate,
        string $brand,
        array $features,
        array &$meta,
    ): int {
        $guards = config('pricing.guards', []);

        $min = isset($guards['min_daily_cents'])
            ? (int) $guards['min_daily_cents']
            : 10000; // $100

        $max = isset($guards['max_daily_cents'])
            ? (int) $guards['max_daily_cents']
            : 100000; // $1,000

        $clamped = min(max($dailyRate, $min), $max);

        $meta['guards'] = [
            'input_daily' => $dailyRate,
            'min'         => $min,
            'max'         => $max,
            'clamped_to'  => $clamped,
        ];

        return $clamped;
    }

    /* ---------------------------------------------------------------------
     | Adjustment stubs (return 1.0 multipliers for now)
     |---------------------------------------------------------------------*/

    /**
     * Lead time adjustment.
     * Example future logic:
     *  - Very last-minute (0–1 days): +10–20%
     *  - Medium (7–30 days): baseline
     *  - Very early (90+ days): small discount
     */
    protected function leadTimeAdjustment(array $features): array
{
    $lead = (int) ($features['lead_time_days'] ?? 0);
    $rules = config('pricing.lead_time_rules', []);

    $chosen = 1.0;
    $matchedRule = null;

    foreach ($rules as $rule) {
        $min = (int) ($rule['min_days'] ?? 0);
        $max = (int) ($rule['max_days'] ?? 0);
        $multiplier = (float) ($rule['multiplier'] ?? 1.0);

        if ($lead >= $min && ($max === 0 || $lead < $max)) {
            $chosen = $multiplier;
            $matchedRule = [
                'min_days'   => $min,
                'max_days'   => $max,
                'multiplier' => $multiplier,
            ];
            break;
        }
    }

    return [
        'multiplier' => $chosen,
        'reason'     => 'lead_time_rules',
        'lead_days'  => $lead,
        'rule'       => $matchedRule,
    ];
}

    /**
     * Seasonality adjustment.
     * Future:
     *  - Summer peak vs shoulder seasons
     *  - Ski season pricing, holidays, etc.
     */
    protected function seasonalityAdjustment(array $features): array
    {
        $month = (int) ($features['pickup_month'] ?? 0);

        return [
            'multiplier' => 1.0,
            'reason'     => 'seasonality_stub',
            'month'      => $month,
        ];
    }

    /**
     * Demand / utilisation adjustment.
     * Future:
     *  - Increase price if fleet utilisation is high for those dates.
     */
    protected function demandAdjustment(array $features): array
    {
        $score = (float) ($features['demand_score'] ?? 0.0);

        return [
            'multiplier' => 1.0,
            'reason'     => 'demand_stub',
            'score'      => $score,
        ];
    }

    /**
     * Local events adjustment.
     * Future:
     *  - Boost price when there are major events in Christchurch / Queenstown.
     */
    protected function eventAdjustment(array $features): array
    {
        $score = (float) ($features['local_events_score'] ?? 0.0);

        return [
            'multiplier' => 1.0,
            'reason'     => 'events_stub',
            'score'      => $score,
        ];
    }

    /**
     * Competitor pricing adjustment.
     * Future:
     *  - Use scraped competitor prices to adjust up/down.
     */
    protected function competitorAdjustment(array $features): array
    {
        $score = (float) ($features['competitor_score'] ?? 0.0);

        return [
            'multiplier' => 1.0,
            'reason'     => 'competitors_stub',
            'score'      => $score,
        ];
    }

    /**
     * AI/ML model adjustment hook.
     *
     * Future:
     *  - Feed $features into an ML model / LLM.
     *  - Return a recommended multiplier or absolute rate.
     *
     * For now, it's a no-op that just records that AI is disabled.
     */
    protected function aiModelAdjustment(array $features, int $currentDailyRate): array
    {
        return [
            'multiplier'         => 1.0,
            'reason'             => 'ai_disabled',
            'model'              => null,
            'suggested_daily'    => null,
            'current_daily_input'=> $currentDailyRate,
        ];
    }
}
