<?php

namespace App\Services;

use Stripe\StripeClient;

class StripePaymentService
{
    public function __construct(private StripeClient $stripe) {}

    /** Create a PaymentIntent for a refundable hold (manual capture) and save card for off-session */
    public function createHoldIntent(array $params): \Stripe\PaymentIntent
    {
        // required: amount_cents, currency, customer_id, metadata (job/deposit ids)
        return $this->stripe->paymentIntents->create([
            'amount' => $params['amount_cents'],           // e.g. 20000
            'currency' => $params['currency'] ?? 'nzd',
            'customer' => $params['customer_id'],          // ensure you create/attach a Customer first
            'payment_method_types' => ['card'],
            'capture_method' => 'manual',                  // HOLD (auth) now, capture later
            'setup_future_usage' => 'off_session',         // save mandate for future off-session charges
            'metadata' => $params['metadata'] ?? [],
        ]);
    }

    /** Capture the hold (full or partial) */
    public function capture(string $paymentIntentId, ?int $amountToCaptureCents = null): \Stripe\PaymentIntent
    {
        return $this->stripe->paymentIntents->capture($paymentIntentId, array_filter([
            'amount_to_capture' => $amountToCaptureCents,
        ]));
    }

    /** Release (cancel) the hold */
    public function release(string $paymentIntentId): \Stripe\PaymentIntent
    {
        return $this->stripe->paymentIntents->cancel($paymentIntentId);
    }

    /** Charge later off-session using saved card */
    public function chargeOffSession(array $params): \Stripe\PaymentIntent
    {
        // required: amount_cents, currency, customer_id; optionally payment_method_id
        return $this->stripe->paymentIntents->create([
            'amount' => $params['amount_cents'],
            'currency' => $params['currency'] ?? 'nzd',
            'customer' => $params['customer_id'],
            'payment_method' => $params['payment_method_id'] ?? null, // if omitted, Stripe will pick default
            'off_session' => true,
            'confirm' => true,
            'metadata' => $params['metadata'] ?? [],
        ]);
    }
}
